functor Ula (
	Bus : BUS_Z80
	) : ULA =
struct
	structure Bus = Bus;
	structure Bitfield = Bus.Bitfield;
	structure Register8 = Bus.Register8;
	structure Register16 = Bus.Register16;
	structure Bit = Bitfield.Bit;
	type ula_state = int * (Register8.reg8 list);
	datatype device_state = DS of (ula_state * (ula_state * Bus.bus_state -> Bus.bus_change * device_state));
	val interrupt_interval = 69888;
	val ula_initial_state : ula_state = (100, List.tabulate (8, fn _ => Register8.fromInt 255));

	fun tick (bs, DS(us, f)) =
		f (us, bs)
	;

	fun keyPos "shift"	= (0, 0) (* row * bit *)
	  | keyPos "z"		= (0, 1)
	  | keyPos "x"		= (0, 2)
	  | keyPos "c"		= (0, 3)
	  | keyPos "v"		= (0, 4)
	  | keyPos "a"		= (1, 0)
	  | keyPos "s"		= (1, 1)
	  | keyPos "d"		= (1, 2)
	  | keyPos "f"		= (1, 3)
	  | keyPos "g"		= (1, 4)
	  | keyPos "q"		= (2, 0)
	  | keyPos "w"		= (2, 1)
	  | keyPos "e"		= (2, 2)
	  | keyPos "r"		= (2, 3)
	  | keyPos "t"		= (2, 4)
	  | keyPos "1"		= (3, 0)
	  | keyPos "2"		= (3, 1)
	  | keyPos "3"		= (3, 2)
	  | keyPos "4"		= (3, 3)
	  | keyPos "5"		= (3, 4)
	  | keyPos "0"		= (4, 0)
	  | keyPos "9"		= (4, 1)
	  | keyPos "8"		= (4, 2)
	  | keyPos "7"		= (4, 3)
	  | keyPos "6"		= (4, 4)
	  | keyPos "p"		= (5, 0)
	  | keyPos "o"		= (5, 1)
	  | keyPos "i"		= (5, 2)
	  | keyPos "u"		= (5, 3)
	  | keyPos "y"		= (5, 4)
	  | keyPos "enter"	= (6, 0)
	  | keyPos "l"		= (6, 1)
	  | keyPos "k"		= (6, 2)
	  | keyPos "j"		= (6, 3)
	  | keyPos "h"		= (6, 4)
	  | keyPos "space"	= (7, 0)
	  | keyPos "sym shft"	= (7, 1)
	  | keyPos "m"		= (7, 2)
	  | keyPos "n"		= (7, 3)
	  | keyPos "b"		= (7, 4)
	  | keyPos _ = raise Fail ("Unknown key");
	;
	fun getRow i l = List.nth (l, i);
	fun putRow 
		(i : int) 
		(r : Register8.reg8)
		(l : Register8.reg8 list)
	:
		(Register8.reg8 list)
	= 
		(List.take (l, i)) @ [r] @ (List.drop (l, i+1));
	
	fun changeKey b key (DS(us,f)) =
		let
			val (intr, keys) = us;
			val (row, bn) = keyPos key;
			val r = getRow row keys;
		in
			DS((intr, putRow row (Register8.putBit bn b r) keys), f)
		end	
	; 

	val pressKey  = changeKey Bit.zero;
	val releaseKey  = changeKey Bit.one;

	fun andKeys r keys =
		let
			fun f n acc =
				if n<0 then 
					acc
				else
					if Register8.isNotSet n r then
						f (n-1) (#1 (Register8.andBits (acc, getRow n keys)))
					else
						f (n-1) acc
			;
		in
			f 7 (Register8.fromInt 255)
		end
	;
	fun asString (DS(us, f)) = "| ULA |";

	fun generate_interrupt
		(us : ula_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val (intr, f) = us;
			val nintr = if (Bus.getCycle bs) = Bus.ASCENDING then intr-1 else intr;
		in
			if Bit.isNotSet (Bus.getIORQ bs) andalso Bit.isNotSet (Bus.getM1 bs) then
				ula_waiting ((nintr,f), bs)
			else
				(Bus.resetINT Bus.no_bus_change, DS((nintr, f), generate_interrupt))
		end
	and
	    ula_waiting 
		(us : ula_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		if (Bus.getCycle bs) = Bus.ASCENDING then
			let
				val (intr, f) = us;
			in
				if intr <=0 then
					generate_interrupt ((interrupt_interval, f), bs)
				else
					(Bus.no_bus_change, DS ((intr-1,f), ula_waiting))
			end
		else
			(Bus.no_bus_change, DS (us, ula_waiting))
	;

	val initial_state : device_state = DS (ula_initial_state, ula_waiting);
end;