functor Spectrum (
	structure Cpu : CPU_Z80; 
	structure Memory : MEMORY_Z80;
	structure Ula : ULA;
	structure Bus : BUS_Z80;

	sharing Cpu.Bus = Bus;
	sharing Memory.Bus = Cpu.Bus;
	sharing Ula.Bus = Memory.Bus;
	) : SPECTRUM =
struct
	structure Bus = Bus;
	structure Bitfield = Bus.Bitfield;
	structure Register8 = Bus.Register8;
	structure Register16 = Bus.Register16;
	structure Bit = Bitfield.Bit;
	type mb_state = (
		Bus.bus_state *
		Cpu.device_state * 
		Memory.device_state * 
		Ula.device_state *
		int *		(* Ticks to screen refresh *)
		int *		(* Next screenshot number *)
		int		(* Screen refresh rate *)
	);
	val initial_state : mb_state =  (
		Bus.initial_state,
		Cpu.initial_state,
		Memory.initial_state, 
		Ula.initial_state,
		5,
		0,
		10000
	);
	exception BUS_CONFLICT of Bus.bus_change;
	fun asString ((bstate, d1, d2, d3, _, _, _) : mb_state) = 
		"DEV1:" ^ (Cpu.asString d1) ^
		"BUS:" ^ (Bus.asString bstate) ^
		"DEV2:" ^ (Memory.asString d2) ^
		"DEV3:" ^ (Ula.asString d3) ^
		"\n"
	;
	fun loadROM filename ((bstate, cs, ms, us, t1, t2, t3) : mb_state)=
		let
			val rom_stream = BinIO.openIn filename;
		in
			Memory.readStream rom_stream 0 16384 ms ;
			BinIO.closeIn rom_stream ;
			(bstate, cs, ms, us, t1, t2, t3)
		end
	;
	fun loadSnapshot filename ((bstate, cs, ms, us, t1, t2, t3) : mb_state) : mb_state =
		let
			val snapshot_stream = BinIO.openIn filename;
			val ncs = Cpu.readStream snapshot_stream;
		in
			(Memory.readStream snapshot_stream 16384 49152 ms;
			BinIO.closeIn snapshot_stream ;
			(bstate, ncs, ms, us, t1, t2, t3))
		end
	;
	fun pressKey k ((bstate, cs, ms, us, t1, t2, t3) : mb_state) : mb_state =
		(bstate, cs, ms, Ula.pressKey k us, t1, t2, t3)
	;

	fun releaseKey k ((bstate, cs, ms, us, t1, t2, t3) : mb_state) : mb_state =
		(bstate, cs, ms, Ula.releaseKey k us, t1, t2, t3)
	;
	
	fun setScreenRefreshRate rr ((bstate, cs, ms, us, t1, t2, t3) : mb_state) : mb_state =
		(bstate, cs, ms, us, if t1>rr then rr else t1, t2, rr)
	;

	fun drawScreen mem num inverse =
		let
			val filename = (Int.toString num) ^ ".ppm";
			val str = TextIO.openOut filename;
			fun out_str s = TextIO.output (str,s);
			fun out_int i = TextIO.output (str, Int.toString i);
			fun color2RGB (c : int) = 
				case c of
(* BLACK *)			  0	=> (0,0,0)
(* BLUE *)			| 1	=> (0,0,100)
(* RED *)			| 2	=> (100, 0, 0)
(* MAGENTA *)			| 3	=> (100, 0, 100)
(* GREEN *)			| 4	=> (0, 100, 0)
(* CYAN *)			| 5	=> (0, 60, 100)
(* YELLOW *)			| 6	=> (0, 100, 100)
(* WHITE *)			| 7	=> (120,120,120)
				| 8	=> (80,80,80)
				| 9	=> (0,0,180)
				| 10	=> (180, 0, 0)
				| 11	=> (180, 0, 180)
				| 12	=> (0, 180, 0)
				| 13	=> (0, 120, 200)
				| 14	=> (0, 180, 180)
				| 15	=> (220,220,220)
				| _	=> raise Fail("Unknown color")
			;
			fun printRGB (c1, c2, c3) =
				(out_int c1 ;
				out_str " ";
				out_int c2;
				out_str " ";
				out_int c3;
				out_str "\n"
				)
			;
			fun chooseColor atr ink inverse =
				let
					val a = Register8.fromInt atr;
					val blink = Bit.asBool (Register8.getBit 7 a);
					val bright = Bit.asBool (Register8.getBit 6 a);
					val i = if ink then 0 else 3;
					val t = if (*blink andalso inverse*) false then
							if i = 0 then 3 else 0
						else
							i
					;
					val bf = Bitfield.subField t 3 (Register8.asBitfield a);
					val g = Bitfield.asUnsigned bf;
				in
					if bright then g+8 else g
				end
			;
			fun draw (x,y) inverse =
				let
					val vidbeg = 16384;
					val atrbeg = vidbeg + ((256 div 8)*192);
					val atrind = atrbeg + ((y div 8)*(256 div 8)) + (x div 8);
					val atr = Array.sub (mem, atrind);
					val bitind = vidbeg + ((256 div 8)*y) + (x div 8);
					val bitofs = x mod 8;
					val bt = Array.sub (mem, bitind);
					val b = Bit.asBool (Register8.getBit bitofs (Register8.fromInt bt));
				in
					printRGB (color2RGB(chooseColor atr b inverse));
					if (x>=255) andalso (y >=191) then 
						()
					else
						if (x >= 255) then
							draw (0, y+1) inverse
						else
							draw (x+1, y) inverse
				end
			;
		in
			out_str "P3\n";
			out_str "256\n";
			out_str "192\n";
			out_str "255\n";
			draw (0, 0) inverse;
			TextIO.closeOut str
		end
	;

	fun refreshScreen ((bstate, cs, ms, us, t1, t2, t3) : mb_state) : mb_state =
		(drawScreen (Memory.getArray ms) t2 (odd t1);
		(bstate, cs, ms, us, t3, t2+1, t3)
		)
	;

	fun tick ((bstate, d1, d2, d3, t, nx, rr) : mb_state) : mb_state = 
		let
			val (b1, d1')  = Cpu.tick (bstate, d1);
			val (b2, d2')  = Memory.tick (bstate, d2);
			val (b3, d3')  = Ula.tick (bstate, d3);
			val l = [b1, b2, b3];
			val (newbus, err) = Bus.checkSignalConflict (l, bstate);
		in
			if isSome(err) then
				raise BUS_CONFLICT(valOf(err))
			else
				if t <= 0 then 
					refreshScreen (newbus, d1', d2', d3', t, nx, rr)
				else
					(newbus, d1', d2', d3', t-1, nx, rr)
		end
	;

end;


