fun odd (x : int) = x mod 2 <> 0;
fun even (x : int) = not (odd x);

functor Register8 (Bitfield : BITFIELD) : REGISTER8 = 
struct
	structure Bitfield = Bitfield;
	structure Bit = Bitfield.Bit;
	exception REGISTER8_ERROR of string;
	type flags = {
		C : Bit.bit option,
		P : Bit.bit option,
		V : Bit.bit option,
		H : Bit.bit option,
		Z : Bit.bit option,
		S : Bit.bit option
	};

	fun getC (f : flags) = #C f;
	fun getP (f : flags) = #P f;
	fun getV (f : flags) = #V f;
	fun getH (f : flags) = #H f;
	fun getZ (f : flags) = #Z f;
	fun getS (f : flags) = #S f;

	val noFlags : flags = {
		C = NONE,
		P = NONE,
		V = NONE,
		H = NONE,
		Z = NONE,
		S = NONE
	}

	type reg8 = Bitfield.bitfield;

	val maxSigned  = 127;
	val maxUnsigned  = 255;
	val minSigned = ~128;
	val minUnsigned  = 0;

	fun setBit (n : int) (r : reg8) : reg8 = Bitfield.setBit n r;
	fun resetBit (n : int) (r : reg8) : reg8  = Bitfield.resetBit n r;
	fun negateBit (n : int) (r : reg8) : reg8  = Bitfield.negateBit n r;

	fun isSet (n : int) (r : reg8) = Bitfield.isSet n r;
	fun isNotSet (n : int) (r : reg8) = Bitfield.isNotSet n r;

	fun asSigned (r : reg8) = Bitfield.asSigned r;
	fun asUnsigned (r : reg8) = Bitfield.asUnsigned r;

	fun fromInt (i : int) : reg8 = Bitfield.fromInt i 8;
	val new  : reg8 = Bitfield.new 8;

	val zero = new;

	fun getBit (n : int) (r : reg8) : Bit.bit = Bitfield.getBit n r;
	fun putBit (n : int) (b : Bit.bit) (r : reg8) : reg8 = Bitfield.putBit n b r;

	fun putC (b : Bit.bit option) (f : flags) : flags =
		{
			C = b,
			P = #P f,
			V = #V f,
			H = #H f,
			Z = #Z f,
			S = #S f
		}
	;
	fun putS (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = #V f,
			H = #H f,
			Z = #Z f,
			S = b
		}
	;

	fun putZ (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = #V f,
			H = #H f,
			Z = b,
			S = #S f
		}
	;

	fun putP (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = b,
			V = #V f,
			H = #H f,
			Z = #Z f,
			S = #S f
		}
	;

	fun putH (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = #V f,
			H = b,
			Z = #Z f,
			S = #S f
		}
	;

	fun putV (b : Bit.bit option) (f : flags) : flags =
		{
			C = #C f,
			P = #P f,
			V = b,
			H = #H f,
			Z = #Z f,
			S = #S f
		}
	;

	val setC = putC (SOME(Bit.one));
	val resetC = putC (SOME(Bit.one));
	fun setCbool (b : bool) = putC (SOME(Bit.fromBool b));
	fun setPbool (b : bool) = putP (SOME(Bit.fromBool b));
	fun setZbool (b : bool) = putZ (SOME(Bit.fromBool b));
	fun setSbool (b : bool) = putS (SOME(Bit.fromBool b));
	fun setHbool (b : bool) = putH (SOME(Bit.fromBool b));
	fun setVbool (b : bool) = putV (SOME(Bit.fromBool b));

	fun setNormalFlags (r : reg8) : flags =
		let
			val v1 = setSbool (isSet 7 r) noFlags;
			val v2 = setZbool ((asUnsigned r) = 0) v1;
			val v3 = setPbool (even (Bitfield.countOnes r)) v2;
		in
			v3
		end
	;


	fun logop2 (r1 : reg8, r2 : reg8) (oper : Bitfield.bitfield * Bitfield.bitfield -> Bitfield.bitfield) : (reg8 * flags) =
		let
			val v : reg8 = oper (r1, r2);
			val f = setNormalFlags v
		in
			(v, f)
		end
	;

	fun logop1 (r: reg8) (oper : Bitfield.bitfield -> Bitfield.bitfield) : (reg8 * flags) =
		let
			val v : reg8 = oper r;
			val f = setNormalFlags v
		in
			(v, f)
		end
	;

	fun andBits (r1 : reg8, r2 : reg8) : (reg8 * flags) = logop2 (r1, r2) Bitfield.andBits;
	fun xorBits (r1 : reg8, r2 : reg8) : (reg8 * flags) = logop2 (r1, r2) Bitfield.xorBits;
	fun orBits (r1 : reg8, r2 : reg8) : (reg8 * flags) = logop2 (r1, r2) Bitfield.orBits;

	fun isHalfCarry (carries : Bitfield.bitfield) = Bitfield.isSet 3 carries;
	fun isHalfBorrow (borrows : Bitfield.bitfield) = Bitfield.isSet 3 borrows;
	fun isOverflow (carries : Bitfield.bitfield) = 
		Bit.asBool (Bit.xorBits (Bitfield.getBit 6 carries, Bitfield.getBit 7 carries))
	;
	fun isCarry (carries : Bitfield.bitfield) = Bitfield.isSet 7 carries;
	fun add (carry : Bit.bit, r1 : reg8, r2 : reg8) : (reg8 * flags) =
		let
			val (v, c) = Bitfield.addBits (carry, r1, r2)
		in
			(v, setHbool (isHalfCarry c) ((setVbool (isOverflow c) (setCbool (isCarry c) (setNormalFlags v)))))
		end
	;

	fun subtract (borrow : Bit.bit, r1 : reg8, r2 : reg8) : (reg8 * flags) =
		let
			val (v, b) = Bitfield.subtractBits (borrow, r1, r2)
		in
			(v, setHbool (isHalfBorrow b) ((setVbool (isOverflow b) (setCbool (not (isCarry b)) (setNormalFlags v)))))
		end
	;
	
	fun negateBits (r : reg8) : (reg8 * flags) = logop1 r Bitfield.negateBits;
	fun u2Bits (r : reg8) : (reg8 * flags) = logop1 r Bitfield.u2Bits;

	fun compare (r1 : reg8, r2 : reg8) : flags =
		let
			val (v, f) = subtract (Bit.zero, r1, r2);
		in
			f
		end
	;

	fun rlBits (newbit : Bit.bit) (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.append (r ,Bitfield.fromBit newbit);
			val r2 = Bitfield.subField 0 8 (Bitfield.rolBits 1 r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rrBits (newbit : Bit.bit) (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.append (r, Bitfield.fromBit newbit);
			val r2 = Bitfield.subField 0 8 (Bitfield.rorBits 1 r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rlcBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r2 = Bitfield.rolBits 1 r;
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun rrcBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r2 = Bitfield.rorBits 1 r;
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun slaBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.subField 0 7 r;
			val r2 = Bitfield.append (Bitfield.fromBit Bit.zero, r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun sllBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 7 r;
			val r1 = Bitfield.subField 0 7 r;
			val r2 = Bitfield.append (Bitfield.fromBit Bit.one, r1);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun sraBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.subField 1 7 r;
			val r2 = Bitfield.append (r1 ,Bitfield.subField 7 1 r);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun srlBits (r : reg8) : (reg8 * flags * Bit.bit) =
		let
			val c = Bitfield.getBit 0 r;
			val r1 = Bitfield.subField 1 7 r;
			val r2 = Bitfield.append (r1, Bitfield.fromBit Bit.zero);
		in
			(r2, setNormalFlags r2, c)
		end
	;

	fun halves (r : reg8) : (Bitfield.bitfield * Bitfield.bitfield) =
		(Bitfield.subField 0 4 r, Bitfield.subField 4 4 r)
	;
	fun fromHalves (bf1 : Bitfield.bitfield , bf2 : Bitfield.bitfield) : reg8 =
		if (Bitfield.width bf1 <>4) orelse (Bitfield.width bf2 <> 4) then
			raise REGISTER8_ERROR("Trying to make reg8 from wrong bitfields")
		else
			Bitfield.append (bf1, bf2)
	;
	fun asString (r : reg8) = (Bitfield.asString r) ^ "(" ^ (intPadded 3 (asUnsigned r)) ^ ")";
	fun fromBitfield (bf : Bitfield.bitfield) : reg8 =
		if Bitfield.width bf <> 8 then
			raise REGISTER8_ERROR("Trying to make reg8 from wrong bitfield")
		else
			bf
	;
	fun asBitfield (r : reg8) : Bitfield.bitfield = r;
end;
