functor CpuZ80 (
	Bus : BUS_Z80
	) : CPU_Z80 =
struct
	structure Bus = Bus;
	structure Bitfield = Bus.Bitfield;
	structure Register8 = Bus.Register8;
	structure Register16 = Bus.Register16;
	structure Bit = Bitfield.Bit;

	val control_stream = TextIO.openOut "stan_cpu.st";
	fun pr s = (TextIO.output (control_stream, s) ; TextIO.flushOut control_stream);

	type reg8 = Register8.reg8; 
	type reg16 = Register16.reg16;

	datatype HLTreating = asHL | asIX | asIY;
	
	exception UNKNOWN_OPCODE of {opcode : reg8 , PC : reg16};
	type cpu_state = {
		A  : reg8,
		A' : reg8,
		F  : reg8,
		F' : reg8,
		B  : reg8,
		B' : reg8,
		C  : reg8,
		C' : reg8,
		D  : reg8,
		D' : reg8,
		E  : reg8,
		E' : reg8,
		H  : reg8,
		H' : reg8,
		L  : reg8,
		L' : reg8,
		I  : reg8,
		R  : reg8,
		IX : reg16,
		IY : reg16,
		SP : reg16,
		PC : reg16,
		(* internal registers *)
		IM : int,	(* Interrupt mode *)
		IFF1 : bool,	(* Interrupts enabled *)
		HLT : bool,	(* Halt state *)
		IgnoreNextInterrupt : bool,
		HLas : HLTreating, (* Treat HL as: HL, IX or IY*)
		inNMI : bool, 
		IFF2 : bool  (* Interrupts state before NMI *)
	};

	val cpu_initial_state : cpu_state = {
		A  = Register8.new,
		A' = Register8.new,
		F  = Register8.new,
		F' = Register8.new,
		B  = Register8.new,
		B' = Register8.new,
		C  = Register8.new,
		C' = Register8.new,
		D  = Register8.new,
		D' = Register8.new,
		E  = Register8.new,
		E' = Register8.new,
		H  = Register8.new,
		H' = Register8.new,
		L  = Register8.new,
		L' = Register8.new,
		I  = Register8.new,
		R  = Register8.new,
		IX = Register16.new,
		IY = Register16.new,
		SP = Register16.new,
		PC = Register16.new,
		(* internal registers *)
		IM = 0,	(* Interrupt mode *)
		IFF1 = true,	(* Interrupts enabled *)
		HLT = false,	(* Halt state *)
		IgnoreNextInterrupt = false,
		HLas = asHL,
		inNMI = false,
		IFF2 = true  (* Interrupts state before NMI *)
		
	};

        datatype device_state = DS of (
		cpu_state
		*
		((cpu_state * Bus.bus_state) -> (Bus.bus_change * device_state))
	);
	type Continuation = cpu_state * Bus.bus_state -> Bus.bus_change * device_state;

	fun getDS (DS(v)) = v;
	fun set_device_id (i:int) (d:device_state) = d;

        fun tick (bs : Bus.bus_state, DS(cpu,f)) =
(*FIXME: tu dodac sprawdzanie na poczatku cyklu maszynowego, czy nie trzeba
odlaczyc sie od magistral *)
		f (cpu, bs)
 	;
	fun cpuAsString (cs : cpu_state) =  
		"|" ^
		"A=" ^ Register8.asString (#A cs) ^
		" F=" ^ Register8.asString (#F cs) ^
		" B=" ^ Register8.asString (#B cs) ^
		" C=" ^ Register8.asString (#C cs) ^
		" D=" ^ Register8.asString (#D cs) ^
		" E=" ^ Register8.asString (#E cs) ^
		" H=" ^ Register8.asString (#H cs) ^
		" L=" ^ Register8.asString (#L cs) ^
		" I=" ^ Register8.asString (#I cs) ^
		" R=" ^ Register8.asString (#R cs) ^
		" IX=" ^ Register16.asString (#IX cs) ^
		" IY=" ^ Register16.asString (#IY cs) ^
		" SP=" ^ Register16.asString (#SP cs) ^
		" PC=" ^ Register16.asString (#PC cs) ^
		" IM=" ^ Int.toString (#IM cs) ^
		" IFF1=" ^ Bool.toString (#IFF1 cs) ^
		" HLT=" ^ Bool.toString (#HLT cs) ^
		" IFF2=" ^ Bool.toString (#IFF2 cs) ^
		" A'=" ^ Register8.asString (#A' cs) ^
		" F'=" ^ Register8.asString (#F' cs) ^
		" B'=" ^ Register8.asString (#B' cs) ^
		" C'=" ^ Register8.asString (#C' cs) ^
		" D'=" ^ Register8.asString (#D' cs) ^
		" E'=" ^ Register8.asString (#E' cs) ^
		" H'=" ^ Register8.asString (#H' cs) ^
		" L'=" ^ Register8.asString (#L' cs) ^
		" IgnoreNextInterrupt=" ^ Bool.toString (#IgnoreNextInterrupt cs) ^
		"|"
	;
	fun asString (DS(cs, _) : device_state) = cpuAsString cs;
	fun putXX v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putA v (cs: cpu_state) : cpu_state = 
	{
		A  = v,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putA' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = v,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putF v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = v,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putF' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = v,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putB v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = v,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putB' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = v,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putC v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = v,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putC' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = v,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putD v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = v,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putD' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = v,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putE v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = v,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putE' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = v,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putH v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = v,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putH' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = v,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putL v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = v,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putL' v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = v,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putI v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = v,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putR v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = v,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putIX v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = v,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putIY v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = v,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putSP v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = v,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putPC v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = v,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putIM v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = v,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putIFF1 v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = v,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putHLT v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = v,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
	fun putHLas v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = v
	};
	fun putIFF2 v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = v,
		inNMI = #inNMI cs,
		HLas = #HLas cs
	};

	fun putInNMI v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
		IFF2 = #IFF2 cs,
		inNMI = v,
		HLas = #HLas cs
	};
	fun putIgnoreNextInterrupt v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IgnoreNextInterrupt = v,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs
	};
	fun putAF (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putF (Register8.fromBitfield r1) (putA (Register8.fromBitfield r2) cs)
		end
	;
	fun putBC (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putC (Register8.fromBitfield r1) (putB (Register8.fromBitfield r2) cs)
		end
	;
	fun putDE (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putE (Register8.fromBitfield r1) (putD (Register8.fromBitfield r2) cs)
		end
	;
	fun putHL (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putL (Register8.fromBitfield r1) (putH (Register8.fromBitfield r2) cs)
		end
	;

	fun putAF' (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putF' (Register8.fromBitfield r1) (putA' (Register8.fromBitfield r2) cs)
		end
	;
	fun putBC' (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putC' (Register8.fromBitfield r1) (putB' (Register8.fromBitfield r2) cs)
		end
	;
	fun putDE' (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putE' (Register8.fromBitfield r1) (putD' (Register8.fromBitfield r2) cs)
		end
	;
	fun putHL' (r : reg16) (cs : cpu_state) = 
		let
			val (r1, r2) = Register16.halves r;
		in
			putL' (Register8.fromBitfield r1) (putH' (Register8.fromBitfield r2) cs)
		end
	;

	fun getAF (cs : cpu_state) : reg16 =
		let
			val v = (Register8.asBitfield (#F cs), Register8.asBitfield (#A cs));
		in
			Register16.fromHalves v
		end
	;

	fun getBC (cs : cpu_state) : reg16 = 
		let
			val v = (Register8.asBitfield (#C cs), Register8.asBitfield (#B cs));
		in
			Register16.fromHalves v
		end
	;


	fun getDE (cs : cpu_state) : reg16 = 
		let
			val v = (Register8.asBitfield (#E cs), Register8.asBitfield (#D cs));
		in
			Register16.fromHalves v
		end
	;

	fun getHL (cs : cpu_state) : reg16 = 
		let
			val v = (Register8.asBitfield (#L cs), Register8.asBitfield (#H cs));
		in
			Register16.fromHalves v
		end
	;

	fun getIYh (cs : cpu_state) : reg8 = 
		let
			val (h1,h2) = Register16.halves (#IY cs);
		in
			Register8.fromBitfield h2
		end
	;

	fun getIYl (cs : cpu_state) : reg8 = 
		let
			val (h1,h2) = Register16.halves (#IY cs);
		in
			Register8.fromBitfield h1
		end
	;

	fun getIXh (cs : cpu_state) : reg8 = 
		let
			val (h1,h2) = Register16.halves (#IX cs);
		in
			Register8.fromBitfield h2
		end
	;

	fun getIXl (cs : cpu_state) : reg8 = 
		let
			val (h1,h2) = Register16.halves (#IX cs);
		in
			Register8.fromBitfield h1
		end
	;

	fun putIYl v (cs : cpu_state) = 
		let
			val (h1,h2) = Register16.halves (#IY cs);
			val nh = Register8.asBitfield v;
		in
			putIY (Register16.fromHalves (nh, h2)) cs
		end
	;

	fun putIYh v (cs : cpu_state) = 
		let
			val (h1,h2) = Register16.halves (#IY cs);
			val nh = Register8.asBitfield v;
		in
			putIY (Register16.fromHalves (h1, nh)) cs
		end
	;

	fun putIXl v (cs : cpu_state) = 
		let
			val (h1,h2) = Register16.halves (#IX cs);
			val nh = Register8.asBitfield v;
		in
			putIX (Register16.fromHalves (nh, h2)) cs
		end
	;

	fun putIXh v (cs : cpu_state) = 
		let
			val (h1,h2) = Register16.halves (#IX cs);
			val nh = Register8.asBitfield v;
		in
			putIX (Register16.fromHalves (h1, nh)) cs
		end
	;

	fun rplPutHL (cs : cpu_state) = 
		case #HLas cs of
		   asHL => putHL
		|  asIX => putIX
		|  asIY => putIY
	;

	fun rplGetHL (cs : cpu_state) = 
		case #HLas cs of
		   asHL => getHL 
		|  asIX => #IX 
		|  asIY => #IY 
	;


	fun rplPutH (cs : cpu_state) = 
		case #HLas cs of
		   asHL => putH
		|  asIX => putIXh
		|  asIY => putIYh
	;

	fun rplPutL (cs : cpu_state) = 
		case #HLas cs of
		   asHL => putL
		|  asIX => putIXl
		|  asIY => putIYl
	;

	fun rplGetH (cs : cpu_state) = 
		case #HLas cs of
		   asHL => #H 
		|  asIX => getIXh 
		|  asIY => getIYh 
	;

	fun rplGetL (cs : cpu_state) = 
		case #HLas cs of
		   asHL => #L 
		|  asIX => getIXl 
		|  asIY => getIYl 
	;

(*	fun putXX v (cs: cpu_state) : cpu_state = 
	{
		A  = #A cs,
		A' = #A' cs,
		F  = #F cs,
		F' = #F' cs,
		B  = #B cs,
		B' = #B' cs,
		C  = #C cs,
		C' = #C' cs,
		D  = #D cs,
		D' = #D' cs,
		E  = #E cs,
		E' = #E' cs,
		H  = #H cs,
		H' = #H' cs,
		L  = #L cs,
		L' = #L' cs,
		I  = #I cs,
		R  = #R cs,
		IX = #IX cs,
		IY = #IY cs,
		SP = #SP cs,
		PC = #PC cs,
		IM = #IM cs,
		IFF1 = #IFF1 cs,
		HLT = #HLT cs,
		IFF2 = #IFF2 cs,
		inNMI = #inNMI cs,
		HLas = #HLas cs 
	};
*)

	fun putFlag_C (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 0 b (#F cs)) cs;
	fun putFlag_N (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 1 b (#F cs)) cs;
	fun putFlag_PV (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 2 b (#F cs)) cs;
	fun putFlag_H (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 4 b (#F cs)) cs;
	fun putFlag_Z (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 6 b (#F cs)) cs;
	fun putFlag_S (b : Bit.bit) (cs : cpu_state) = putF (Register8.putBit 7 b (#F cs)) cs;

	fun getFlag_C (cs : cpu_state) = Register8.getBit 0 (#F cs);
	fun getFlag_PV (cs : cpu_state) = Register8.getBit 2 (#F cs);
	fun getFlag_Z (cs : cpu_state) = Register8.getBit 6 (#F cs);
	fun getFlag_S (cs : cpu_state) = Register8.getBit 7 (#F cs);

	val resetIFF1 = putIFF1 false;
	val resetIFF2 = putIFF2 false;
	val setIFF1 = putIFF1 true;
	val setIFF2 = putIFF2 true;

	val resetInNMI = putInNMI false;
	val setInNMI = putInNMI true;

	val resetFlag_C = putFlag_C Bit.zero;
	val setFlag_C = putFlag_C Bit.one;
	val resetFlag_N = putFlag_N Bit.zero;
	val setFlag_N = putFlag_N Bit.one;
	val resetFlag_H = putFlag_H Bit.zero;
	val setFlag_H = putFlag_H Bit.one;
	val resetFlag_Z = putFlag_Z Bit.zero;
	val setFlag_Z = putFlag_Z Bit.one;
	val resetFlag_PV = putFlag_PV Bit.zero;
	val setFlag_PV = putFlag_PV Bit.one;

	fun id x = x;
	fun convC (f : Register8.flags) = if isSome (Register8.getC f) then putFlag_C (valOf (Register8.getC f)) else id;
	fun convV (f : Register8.flags) = if isSome (Register8.getV f) then putFlag_PV (valOf (Register8.getV f)) else id;
	fun convP (f : Register8.flags) = if isSome (Register8.getP f) then putFlag_PV (valOf (Register8.getP f)) else id;
	fun convH (f : Register8.flags) = if isSome (Register8.getH f) then putFlag_H (valOf (Register8.getH f)) else id;
	fun convZ (f : Register8.flags) = if isSome (Register8.getZ f) then putFlag_Z (valOf (Register8.getZ f)) else id;
	fun convS (f : Register8.flags) = if isSome (Register8.getS f) then putFlag_S (valOf (Register8.getS f)) else id;

	fun conv16C (f : Register16.flags) = if isSome (Register16.getC f) then putFlag_C (valOf (Register16.getC f)) else id;
	fun conv16V (f : Register16.flags) = if isSome (Register16.getV f) then putFlag_PV (valOf (Register16.getV f)) else id;
	fun conv16P (f : Register16.flags) = if isSome (Register16.getP f) then putFlag_PV (valOf (Register16.getP f)) else id;
	fun conv16Z (f : Register16.flags) = if isSome (Register16.getZ f) then putFlag_Z (valOf (Register16.getZ f)) else id;
	fun conv16S (f : Register16.flags) = if isSome (Register16.getS f) then putFlag_S (valOf (Register16.getS f)) else id;

	(* Auxiliary functions *)

	fun unknown_opcode_string (opcode) (cs : cpu_state) =
		"* UNKNOWN OPCODE: " ^
		 Int.toString opcode ^
		" at address " ^
		Int.toString (Register16.asUnsigned (#PC cs)) ^
		"\n"
	;

	fun prPC (cs : cpu_state) = pr (intPadded 3 (Register16.asUnsigned (#PC cs)));

	fun add2Adr i (r : Register16.reg16) = #1 (Register16.add (Bit.zero, r, Register16.fromInt i));
	val incAdr = add2Adr 1;
	val decAdr = add2Adr ~1;

	fun add2Reg i (r : Register8.reg8) = #1 (Register8.add (Bit.zero, r, Register8.fromInt i));
	val incReg = add2Reg 1;
	val decReg = add2Reg ~1;


	fun incPC (cs : cpu_state) = putPC (incAdr (#PC cs)) cs;
	fun decPC (cs : cpu_state) = putPC (decAdr (#PC cs)) cs;
	fun incSP (cs : cpu_state) = putSP (incAdr (#SP cs)) cs;
	fun decSP (cs : cpu_state) = putSP (decAdr (#SP cs)) cs;
	fun decBC (cs : cpu_state) = putBC (decAdr (getBC cs)) cs;
	fun decHL (cs : cpu_state) = putHL (decAdr (getHL cs)) cs;
	fun decDE (cs : cpu_state) = putDE (decAdr (getDE cs)) cs;
	fun incBC (cs : cpu_state) = putBC (incAdr (getBC cs)) cs;
	fun incHL (cs : cpu_state) = putHL (incAdr (getHL cs)) cs;
	fun incDE (cs : cpu_state) = putDE (incAdr (getDE cs)) cs;

	fun decB (cs : cpu_state) = putB (decReg (#B cs)) cs;
	
	fun joinRegs (r1, r2) = Register16.fromHalves (Register8.asBitfield r1, Register8.asBitfield r2);
	fun splitRegs r = 
		let
			val (t1, t2) = Register16.halves r;
			val r1 = Register8.fromBitfield t1;
			val r2 = Register8.fromBitfield t2;
		in
			(r1, r2)
		end
	;

	(* The core - state changes *)

	fun next_instruction (cs : cpu_state, bs : Bus.bus_state) =
		let
			val ncs = ((putHLas asHL) o (putIgnoreNextInterrupt false)) cs;
		in
		if (Bus.getCycle bs) = Bus.ASCENDING then
			if Bit.isNotSet(Bus.getNMI bs) andalso not (#inNMI cs) then
				acceptNMI (cs,bs)
			else
				if Bit.isNotSet (Bus.getINT bs) andalso (#IFF1 cs) andalso not (#IgnoreNextInterrupt cs) then
					acceptINT (ncs,bs)
				else
					oOCF choose_opcode (ncs, bs)
		else
			(Bus.no_bus_change, DS(cs, next_instruction))
		end
	and
	   acceptINT
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			fun f data
				(cs : cpu_state, bs : Bus.bus_state)
			=
				case #IM cs of
				  0 => choose_opcode data (cs, bs)
				| 1 => op_RST_adr 56 (cs, bs)
				| 2 => 	let
						val vecadr = Register16.fromHalves (Register8.asBitfield data, Register8.asBitfield (#I cs));
						fun h jumpadr
							(cs : cpu_state, bs : Bus.bus_state)
						=
							(Bus.no_bus_change, DS(putPC jumpadr cs, next_instruction))
						;
						fun g
							(cs : cpu_state, bs : Bus.bus_state)
						=
							oMR16 h vecadr (cs, bs)
						;
					in
						oSW16 6 g (#PC cs) (cs, bs)
					end
				| _ => raise Fail("Wrong interrupt mode")
			;
		in
			(pr "INTRRUPT\n" ; accept_int f (#PC cs) ((resetIFF1 o resetIFF2) cs, bs))
		end
	and
	   acceptNMI
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			fun f 
				(c : cpu_state, b : Bus.bus_state)
			=
				(* 66h = 102 *)
				oSW16 6 next_instruction (#PC cs (*intentional*)) (putPC (Register16.fromInt 102) c, b)
			;
		in
			read_opcode (fn _ => pad_ticks 1 f) (#PC cs) ((resetIFF1 o setInNMI) cs, bs)
		end
	and
	    read_arg8
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		read_memory cont adr (cs, bs)
	and
	    do_read_memory
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val T1_waiting_with_address =
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 1 Register8.zero)
					)
				);
			val T1_waiting_with_MREQ =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 1 Register8.zero)
					)
				);
			val T2_waiting_with_MREQ =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 2 Register8.zero)
					)
				);
			val T2_waiting_for_reading =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 2 Register8.zero)
					)
				);
			val T3_waiting_for_reading =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 3 Register8.zero)
					)
				);
			val T3_waiting_to_pass_data = fn (d : Register8.reg8) =>
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_memory cont adr 3 d)
					)
				);
			val T3_passing_data =
				(
					Bus.no_bus_change
				,
					DS(
						cs
					,
						cont data
					)
				);
		in
			case (step, Bus.getCycle bs) of
				(1, Bus.ASCENDING)	=> T1_waiting_with_address
			|	(1, Bus.HIGH)		=> T1_waiting_with_address
			|	(1, Bus.DESCENDING)	=> (print "\nT1\n" ; T1_waiting_with_MREQ)
			|	(1, Bus.LOW)		=> T2_waiting_with_MREQ
			|	(2, Bus.ASCENDING)	=> (print "\nT2\n" ; T2_waiting_with_MREQ)
			|	(2, Bus.HIGH)		=> T2_waiting_with_MREQ
			|	(2, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then T1_waiting_with_MREQ else T2_waiting_for_reading
			|	(2, Bus.LOW)		=> T3_waiting_for_reading
			|	(3, Bus.ASCENDING)	=> T3_waiting_for_reading
			|	(3, Bus.HIGH)		=> (print "\nT3\n" ; T3_waiting_for_reading)
			|	(3, Bus.DESCENDING)	=> (print "\nT4\n" ; T3_waiting_to_pass_data (Bus.getData bs))
			|	(3, Bus.LOW)		=> cont data (cs, bs)
			|	_ => raise Fail("cpuZ80: Unknown state during reading memory")
		end
	and

	    read_memory
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
(*			raise Fail("Trying to read memory begining from other than ASCENDING state") *)
			(Bus.no_bus_change, DS(cs, read_memory cont adr))
		else
			do_read_memory cont adr 1 Register8.zero (cs, bs) 
	and
	    do_accept_int
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val Ti_wait = fn i =>
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_accept_int cont adr i Register8.zero)
					)
				);
			val Ti_wait2 = fn i =>
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					o Bus.resetIORQ
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_accept_int cont adr i Register8.zero)
					)
				);
			val Ti_wait3 = fn i => fn dat =>
				(
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_accept_int cont adr i dat)
					)
				);
		in
			(* FIXME - no RFSH *)
			case (step, Bus.getCycle bs) of
				(0, Bus.ASCENDING)	=> Ti_wait 0
			|	(0, Bus.HIGH)		=> Ti_wait 0
			|	(0, Bus.DESCENDING)	=> Ti_wait 0
			|	(0, Bus.LOW)		=> Ti_wait 1
			|	(1, Bus.ASCENDING)	=> Ti_wait 1
			|	(1, Bus.HIGH)		=> Ti_wait 1
			|	(1, Bus.DESCENDING)	=> Ti_wait 1
			|	(1, Bus.LOW)		=> Ti_wait 2
			|	(2, Bus.ASCENDING)	=> Ti_wait 2
			|	(2, Bus.HIGH)		=> Ti_wait 2
			|	(2, Bus.DESCENDING)	=> Ti_wait2 2
			|	(2, Bus.LOW)		=> Ti_wait2 3
			|	(3, Bus.ASCENDING)	=> Ti_wait2 3
			|	(3, Bus.HIGH)		=> Ti_wait2 3
			|	(3, Bus.DESCENDING)	=> Ti_wait2 3
			|	(3, Bus.LOW)		=> Ti_wait2 4
			|	(4, Bus.ASCENDING)	=> Ti_wait3 4 data
			|	(4, Bus.HIGH)		=> Ti_wait3 4 data
			|	(4, Bus.DESCENDING)	=> Ti_wait3 4 data
			|	(4, Bus.LOW)		=> Ti_wait3 5 data
			|	(5, Bus.ASCENDING)	=> Ti_wait3 5 data
			|	(5, Bus.HIGH)		=> Ti_wait3 5 data
			|	(5, Bus.DESCENDING)	=> Ti_wait3 5 data
			|	(5, Bus.LOW)		=> cont data (cs, bs)
			|	_ => raise Fail("cpuZ80: Unknown state during accepting int")
		end
	and

	    accept_int
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
(*			raise Fail("Trying to read port begining from other than ASCENDING state") *)
			(Bus.no_bus_change, DS(cs, accept_int cont adr))
		else
			do_accept_int cont adr 0 Register8.zero (cs, bs) 
	and
	    do_read_port
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val T0_waiting_with_address =
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 0 Register8.zero)
					)
				);
			val T1_waiting_with_address =
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 1 Register8.zero)
					)
				);
			val T1_waiting_with_IORQ =
				(
					((Bus.putAddress adr)
					o Bus.resetIORQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 1 Register8.zero)
					)
				);
			val T2_waiting_with_IORQ =
				(
					((Bus.putAddress adr)
					o Bus.resetIORQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 2 Register8.zero)
					)
				);
			val T3_waiting_for_reading =
				(
					((Bus.putAddress adr)
					o Bus.resetIORQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 3 Register8.zero)
					)
				);
			val T3_waiting_to_pass_data = fn (d : Register8.reg8) =>
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_port cont adr 3 d)
					)
				);
		in
			case (step, Bus.getCycle bs) of
				(0, Bus.ASCENDING)	=> T0_waiting_with_address
			|	(0, Bus.HIGH)		=> T0_waiting_with_address
			|	(0, Bus.DESCENDING)	=> T0_waiting_with_address
			|	(0, Bus.LOW)		=> T1_waiting_with_address
			|	(1, Bus.ASCENDING)	=> T1_waiting_with_IORQ
			|	(1, Bus.HIGH)		=> T1_waiting_with_IORQ
			|	(1, Bus.DESCENDING)	=> T1_waiting_with_IORQ
			|	(1, Bus.LOW)		=> T2_waiting_with_IORQ
			|	(2, Bus.ASCENDING)	=> T2_waiting_with_IORQ
			|	(2, Bus.HIGH)		=> T2_waiting_with_IORQ
			|	(2, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then T1_waiting_with_IORQ else T2_waiting_with_IORQ
			|	(2, Bus.LOW)		=> T3_waiting_for_reading
			|	(3, Bus.ASCENDING)	=> T3_waiting_for_reading
			|	(3, Bus.HIGH)		=> T3_waiting_for_reading
			|	(3, Bus.DESCENDING)	=> T3_waiting_to_pass_data (Bus.getData bs)
			|	(3, Bus.LOW)		=> cont data (cs, bs)
			|	_ => raise Fail("cpuZ80: Unknown state during reading port")
		end
	and

	    read_port
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
(*			raise Fail("Trying to read port begining from other than ASCENDING state") *)
			(Bus.no_bus_change, DS(cs, read_port cont adr))
		else
			do_read_port cont adr 0 Register8.zero (cs, bs) 
	and
	    do_write_port
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val T0_waiting_with_address =
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 0 data)
					)
				);
			val T0_waiting_with_data =
				(
					((Bus.putAddress adr)
					o Bus.putData data
					)Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 0 data)
					)
				);
			val T1_waiting_with_data =
				(
					((Bus.putAddress adr)
					o Bus.putData data
					)Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 1 data)
					)
				);
			val T1_waiting_with_IORQ =
				(
					((Bus.putAddress adr)
					o (Bus.putData data)
					o Bus.resetWR
					o Bus.resetIORQ
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 1 data)
					)
				);
			val T2_waiting_with_IORQ =
				(
					((Bus.putAddress adr)
					o (Bus.putData data)
					o Bus.resetIORQ
					o Bus.resetWR
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 2 data)
					)
				);
			val T3_waiting_for_writeing =
				(
					((Bus.putAddress adr)
					o (Bus.putData data)
					o Bus.resetIORQ
					o Bus.resetWR
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 3 data)
					)
				);
			val T3_waiting_to_pass_data =
				(
					((Bus.putAddress adr)
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_port cont adr 3 data)
					)
				);
		in
			case (step, Bus.getCycle bs) of
				(0, Bus.ASCENDING)	=> T0_waiting_with_address
			|	(0, Bus.HIGH)		=> T0_waiting_with_address
			|	(0, Bus.DESCENDING)	=> T0_waiting_with_data
			|	(0, Bus.LOW)		=> T1_waiting_with_data
			|	(1, Bus.ASCENDING)	=> T1_waiting_with_IORQ
			|	(1, Bus.HIGH)		=> T1_waiting_with_IORQ
			|	(1, Bus.DESCENDING)	=> T1_waiting_with_IORQ
			|	(1, Bus.LOW)		=> T2_waiting_with_IORQ
			|	(2, Bus.ASCENDING)	=> T2_waiting_with_IORQ
			|	(2, Bus.HIGH)		=> T2_waiting_with_IORQ
			|	(2, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then T1_waiting_with_IORQ else T2_waiting_with_IORQ
			|	(2, Bus.LOW)		=> T3_waiting_for_writeing
			|	(3, Bus.ASCENDING)	=> T3_waiting_for_writeing
			|	(3, Bus.HIGH)		=> T3_waiting_for_writeing
			|	(3, Bus.DESCENDING)	=> T3_waiting_to_pass_data
			|	(3, Bus.LOW)		=> cont (cs, bs)
			|	_ => raise Fail("cpuZ80: Unknown state during writeing port")
		end
	and

	    write_port
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		data
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
(*			raise Fail("Trying to write port begining from other than ASCENDING state") *)
			(Bus.no_bus_change, DS(cs, write_port cont adr data))
		else
			do_write_port cont adr 0 data (cs, bs) 
	and
	    acceptDMA
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		if (Bit.isNotSet (Bus.getBUSRQ bs)) then
			(Bus.resetBUSAK Bus.no_bus_change, DS(cs, acceptDMA cont))
		else
			if Bus.getCycle bs = Bus.LOW then
				cont (cs, bs)
			else
				(Bus.no_bus_change, DS(cs, acceptDMA cont))
	and
	    pad_ticks
		(ticks : int)
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
	(* 
	   Pads a given number of ticks (not bus changes) till the end of machine cycle.
	   If at the end of cycle there is a DMA request, accepts it.
	   WARNING! cont is executed _after_ acceptation of DMA
	*)
		if ticks < 0 then 
			raise Fail("pad_ticks: ticks<0")
		else
			if ticks = 0 andalso (Bus.getCycle bs) = Bus.LOW then
				if (Bit.isNotSet (Bus.getBUSRQ bs)) then
					acceptDMA (pad_ticks 0 cont) (cs, bs)
				else
					cont  (cs, bs)
			else
				if (Bus.getCycle bs) = Bus.LOW then
					(Bus.no_bus_change, DS(cs, pad_ticks (ticks-1) cont))
				else
					(Bus.no_bus_change, DS(cs, pad_ticks ticks cont))
	and
	    oMR
		(ticks : int)
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		read_memory (fn r => pad_ticks (ticks-3) (cont r)) adr (cs, bs)
	and
	    oPR
		(ticks : int)
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		read_port (fn r => pad_ticks (ticks-4) (cont r)) adr (cs, bs)
	and
	    read_2arg8
		(cont : Register8.reg8 -> Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			fun f 
				(r1 : Register8.reg8) (cs : cpu_state, bs : Bus.bus_state)
			: 
				Bus.bus_change * device_state
			=
				read_memory (fn r2 => pad_ticks 0 (cont r1 r2)) (incAdr adr) (cs, bs)
			;
		in
			read_memory (fn r => pad_ticks 0 (f r)) adr (cs, bs)
		end
	and

	    oMR16
		(cont : Register16.reg16 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		read_2arg8 (fn r1 => fn r2 => cont (Register16.fromHalves (Register8.asBitfield r1, Register8.asBitfield r2))) adr (cs, bs)
	and
	    oODR16
		(cont : Register16.reg16 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val adr = #PC cs;
			fun incPC2 (c : cpu_state) = putPC (add2Adr 2 (#PC c)) c;
		in
			oMR16 (fn r => fn (c,b) => cont r (incPC2 c, b)) adr (cs, bs)
		end
	and
	    oSR16
		(cont : Register16.reg16 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val adr = (#SP cs);
			val ncs = (incSP o incSP) cs; 
		in
			oMR16 (fn r => cont r) adr (ncs, bs)
		end
	and
	    oMW
		(ticks : int)
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		write_memory (pad_ticks (ticks-3) cont) adr data (cs, bs)
	and
	    oPW
		(ticks : int)
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		write_port (pad_ticks (ticks-4) cont) adr data (cs, bs)
	and
	    oSW16
		(ticks : int)
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(data : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val (r1, r2) = splitRegs data;
			fun c (cs : cpu_state, bs : Bus.bus_state) = oMW (ticks - 3) cont (decAdr (#SP cs)) r1 (decSP cs, bs);
		in
			oMW 3 c (decAdr (#SP cs)) r2 (decSP cs, bs)
		end
	and
	    oMW16
		(ticks : int)
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(data : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val (r1, r2) = splitRegs data;
			fun c (cs, bs) = oMW (ticks-3) cont (incAdr adr) r2 (cs, bs);
		in
			oMW 3 c adr r1 (cs, bs)
		end
	and
	    oODR
		(ticks : int)
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val adr = #PC cs;
			val ncs = putPC (incAdr (#PC cs)) cs;
		in
			read_memory (fn r => pad_ticks (ticks-3) (cont r)) adr (ncs, bs)
		end
	and
	    oOCF
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val adr = #PC cs;
			val rc = Register8.getBit 7 (#R cs);
			val (nr,_) = Register8.add (Bit.zero, #R cs, Register8.fromInt 1);
			val nnr = Register8.putBit 7 rc nr;
			val ncs = ((putPC (incAdr (#PC cs))) o (putR nnr)) cs;
		in
			read_opcode (fn r => pad_ticks 0 (cont r)) adr (ncs, bs)
		end
	and
	    do_read_opcode
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
	(* FIXME : no RFSH *)
		let
			val T1_waiting_with_address =
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					)Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 1 Register8.zero)
					)
				);
			val T1_waiting_with_MREQ =
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 1 Register8.zero)
					)
				);
			val T2_waiting_with_MREQ =
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 2 Register8.zero)
					)
				);
			val T2_waiting =
				(
					((Bus.putAddress adr)
					o Bus.resetM1
					o Bus.resetMREQ
					o Bus.resetRD
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 2 data)
					)
				);
			val T3_waiting = fn d =>
				(
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 3 d)
					)
				);
			val T4_waiting =
				(
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_read_opcode cont adr 4 data)
					)
				);
			val T4_passing_data =
				(
					Bus.no_bus_change
				,
					DS(
						cs
					,
						cont data
					)
				);
		in
			case (step, Bus.getCycle bs) of
				(1, Bus.ASCENDING)	=> T1_waiting_with_address
			|	(1, Bus.HIGH)		=> T1_waiting_with_address
			|	(1, Bus.DESCENDING)	=> T1_waiting_with_MREQ
			|	(1, Bus.LOW)		=> T2_waiting_with_MREQ
			|	(2, Bus.ASCENDING)	=> T2_waiting_with_MREQ
			|	(2, Bus.HIGH)		=> T2_waiting_with_MREQ
			|	(2, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then T1_waiting_with_MREQ else T2_waiting
			|	(2, Bus.LOW)		=> T3_waiting (Bus.getData bs)
			|	(3, Bus.ASCENDING)	=> T3_waiting data
			|	(3, Bus.HIGH)		=> T3_waiting data
			|	(3, Bus.DESCENDING)	=> T3_waiting data
			|	(3, Bus.LOW)		=> T4_waiting
			|	(4, Bus.ASCENDING)	=> T4_waiting 
			|	(4, Bus.HIGH)		=> T4_waiting 
			|	(4, Bus.DESCENDING)	=> T4_waiting 
			|	(4, Bus.LOW)		=> T4_passing_data
			|	_ => raise Fail("cpuZ80: Unknown state during reading opcode")
		end
	and

	    read_opcode
		(cont : Register8.reg8 -> cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
			(*raise Fail("Trying to read opcode begining from other than ASCENDING state") *)
			(Bus.no_bus_change, DS(cs, read_opcode cont adr))
		else
			do_read_opcode cont adr 1 Register8.zero (cs, bs) 
	and
	    do_write_memory
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(step : int)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		let
			val T1_waiting_with_address =
				(
					(Bus.putAddress adr)
					Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 1 data)
					)
				);
			val T1_waiting_with_data =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 1 data)
					)
				);
			val T2_waiting_with_data =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 2 data)
					)
				);
			val T2_wrote =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetWR
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 2 data)
					)
				);
			val TW_waiting =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetWR
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 3 data)
					)
				);
			val T3_waiting_with_WR =
				(
					((Bus.putAddress adr)
					o Bus.resetMREQ
					o Bus.resetWR
					o (Bus.putData data)
					) Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 4 data)
					)
				);
			val T3_waiting =
				(
					((Bus.putAddress adr)
					o (Bus.putData data)
					)Bus.no_bus_change
				,
					DS(
						cs
					,
						(do_write_memory cont adr 4 data)
					)
				);
		in
			case (step, Bus.getCycle bs) of
				(1, Bus.ASCENDING)	=> T1_waiting_with_address
			|	(1, Bus.HIGH)		=> T1_waiting_with_address
			|	(1, Bus.DESCENDING)	=> T1_waiting_with_data
			|	(1, Bus.LOW)		=> T2_waiting_with_data
			|	(2, Bus.ASCENDING)	=> T2_waiting_with_data
			|	(2, Bus.HIGH)		=> T2_waiting_with_data
			|	(2, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then TW_waiting else T2_wrote
			|	(2, Bus.LOW)		=> T3_waiting_with_WR
				(* TW - Wait state *)
			|	(3, Bus.ASCENDING)	=> TW_waiting
			|	(3, Bus.HIGH)		=> TW_waiting
			|	(3, Bus.DESCENDING)	=> if Bit.isSet (Bus.getWAIT bs) then TW_waiting else T2_wrote
			|	(3, Bus.LOW)		=> TW_waiting
				(* T3 *)
			|	(4, Bus.ASCENDING)	=> T3_waiting_with_WR
			|	(4, Bus.HIGH)		=> T3_waiting_with_WR
			|	(4, Bus.DESCENDING)	=> T3_waiting
			|	(4, Bus.LOW)		=> cont (cs, bs)
			|	_ => raise Fail("cpuZ80: Unknown state during writing memory")
		end
	and

	    write_memory
		(cont : cpu_state * Bus.bus_state -> Bus.bus_change * device_state)
		(adr : Register16.reg16)
		(data : Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	:
		(Bus.bus_change * device_state)
	=
		(* Assertion: Beginning in ASCENDING cycle state *)
		if (Bus.getCycle bs) <> Bus.ASCENDING then 
			(*raise Fail("Trying to write memory begining from other than ASCENDING state")*)
			(Bus.no_bus_change, DS(cs, write_memory cont adr data))
		else
			do_write_memory cont adr 1 data (cs, bs) 
	and
	    op_NOP
		(cs : cpu_state, bs : Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		pad_ticks 0 next_instruction (cs, bs)
	and
	    op_IM
		i
		(cs : cpu_state, bs : Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		pad_ticks 0 next_instruction (putIM i cs, bs)
	and
	    op_DI
		(cs : cpu_state, bs : Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		pad_ticks 0 next_instruction ((resetIFF1 o resetIFF2 o (putIgnoreNextInterrupt true)) cs, bs)
	and
	    op_EI
		(cs : cpu_state, bs : Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		pad_ticks 0 next_instruction ((setIFF1 o setIFF2 o (putIgnoreNextInterrupt true)) cs, bs)
	and
	    op_LD_r_r
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = get_r2 cs;
					val ncs = (put_r1 r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_LD_r_specr
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = get_r2 cs;
					val (_, f) = Register8.add (Bit.zero, r, Register8.zero);
					val fl = (putFlag_PV (Bit.fromBool (#IFF2 cs))) o resetFlag_N o resetFlag_H o (convZ f) o (convS f);
					val ncs = (fl o (put_r1 r)) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_LD_rr_rr
		(put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register16.reg16)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = get_r2 cs;
					val ncs = (put_r1 r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 2 f
		end
	and
	    op_LD_r_n
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = (put_r1 r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_LD_r_nn
		(put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = (put_r1 r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oODR16 f)
		end
	and
	    op_LD_r_ptr
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(adr : Register16.reg16)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = (put_r1 r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oMR 3 f adr)
		end
	and
	    op_LD_r_ptrnn
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		let
			fun g adr = 
				let
					fun f r (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val ncs = (put_r1 r) cs;
						in
							(Bus.no_bus_change, DS(ncs, next_instruction))
						end
					;
				in
					oMR 3 f adr
				end
			;
		in
			pad_ticks 0 (oODR16 g)
		end
	and
	    op_IN_A_ptrn
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun g arg = 
				let
					fun f r (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val ncs = (putA r) cs;
						in
							(Bus.no_bus_change, DS(ncs, next_instruction))
						end
					;
					val adr = Register16.fromHalves(Register8.asBitfield arg, Register8.asBitfield (#A cs));
				in
					oPR 4 f adr
				end
			;
		in
			pad_ticks 0 (oODR 3 g) (cs, bs)
		end
	and
	    op_IN_r_C
		(put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (_, f) = Register8.add (Bit.zero, r, Register8.zero);
					val fl = resetFlag_N o (convP f) o resetFlag_H o (convZ f) o (convS f);
					val ncs = (fl o (put_r1 r)) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oPR 4 f (getBC cs)) (cs, bs)
		end
	and
	    op_LD_ptr_r
		(get_r1 : cpu_state -> Register8.reg8)
		(adr : Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		pad_ticks 0 (oMW 3 next_instruction adr (get_r1 cs)) (cs, bs)
	and
	    op_LDxx
		(increase : bool)
		(repeat : bool)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let 
			fun f
				(ptrval)
				(cs : cpu_state, bs : Bus.bus_state)
			=
				let
					val decbczero = (getBC (decBC cs) = Register16.zero);
					val bcf = if decbczero then resetFlag_PV else setFlag_PV;
					val fl = (resetFlag_N o bcf o resetFlag_H);
					val de = if increase then incDE else decDE;
					val hl = if increase then incHL else decHL;
					val pc = if repeat andalso (not decbczero) then 
							(decPC o decPC)
						else 
							id
					;
					val ncs = (decBC o de o hl o fl o pc) cs;
				in
					oMW 3 next_instruction (getDE cs) ptrval (ncs, bs)
				end
		in
			oMR 3 f (getHL cs) (cs, bs)
		end
	and
	    op_CPxx
		(increase : bool)
		(repeat : bool)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let 
			fun f
				(ptrval)
				(cs : cpu_state, bs : Bus.bus_state)
			=
				let
					val (_, f) = Register8.subtract (Bit.zero, #A cs, ptrval);
					val decbczero = (getBC (decBC cs) = Register16.zero);
					val bcf = if decbczero then resetFlag_PV else setFlag_PV;
					val fl = (setFlag_N) o bcf o (convH f) o (convZ f) o (convS f);
					val hl = if increase then incHL else decHL;
					val pc = if repeat andalso (not decbczero) andalso Bit.isNotSet (getFlag_Z (fl cs)) then
							(decPC o decPC)
						else 
							id
					;
					val ncs = (decBC o hl o fl o pc) cs;
				in
					pad_ticks 0 next_instruction (ncs, bs)
				end
		in
			oMR 3 f (getHL cs) (cs, bs)
		end
	and
	    op_LD_ptrnn_r
		(get_r1 : cpu_state -> Register8.reg8)
	=
		let
			fun f adr (cs : cpu_state, bs : Bus.bus_state) = 
				oMW 3 next_instruction adr (get_r1 cs) (cs, bs)
		in
			pad_ticks 0 (oODR16 f)
		end
	and
	    op_OUT_ptrn_A
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f arg (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val adr = Register16.fromHalves(Register8.asBitfield arg, Register8.asBitfield (#A cs));
				in
					oPW 4 next_instruction adr (#A cs) (cs, bs)
				end
		in
			pad_ticks 0 (oODR 3 f) (cs, bs)
		end
	and
	    op_OUT_C_r
		(get_r1 : cpu_state -> Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		pad_ticks 0 (oPW 4 next_instruction (getBC cs) (get_r1 cs)) (cs, bs)
	and
	    op_LD_ptr_n
		(adr : Register16.reg16)
	=
		let
			fun f data (cs : cpu_state, bs : Bus.bus_state) = 
				oMW 3 next_instruction adr data (cs, bs)
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_LD_ptrnn_rr
		(get_r1 : cpu_state -> Register16.reg16)
	=
		let
			fun f adr (cs : cpu_state, bs : Bus.bus_state) = 
				oMW16 6 next_instruction adr (get_r1 cs) (cs, bs)
		in
			pad_ticks 0 (oODR16 f)
		end
	and
	    op_LD_rr_ptrnn
		(put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun g adr = 
				let
					fun f r (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val ncs = (put_r1 r) cs;
						in
							(Bus.no_bus_change, DS(ncs, next_instruction))
						end
				in
					oMR16 f adr
				end
			;
		in
			pad_ticks 0 (oODR16 g)
		end
	and
	    op_ADx_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if carry then getFlag_C cs else Bit.zero;
					val (r, f) = Register8.add (cc, get_r1 cs, ptrval);
					val fl = (resetFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_ADC_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_ADx_r_n (get_r1, put_r1) true
	and
	    op_ADD_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_ADx_r_n (get_r1, put_r1) false
	and
	    op_SBx_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if carry then getFlag_C cs else Bit.zero;
					val (r, f) = Register8.subtract (cc, get_r1 cs, ptrval);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_SBC_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_SBx_r_n (get_r1, put_r1) true
	and
	    op_SUB_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_SBx_r_n (get_r1, put_r1) false
	and
	    op_ADx_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
		(carry : bool)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if carry then getFlag_C cs else Bit.zero;
					val (r, f) = Register8.add (cc, get_r1 cs, get_r2 cs);
					val fl = (resetFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_ADC_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_ADx_r_r (get_r1, put_r1) get_r2 true
	and
	    op_ADD_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_ADx_r_r (get_r1, put_r1) get_r2 false
	and
	    op_ADx_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
		(carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.add (getFlag_C cs, get_r1 cs, ptrval);
					val fl = (resetFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oMR 3 f ptr)
		end
	and
	    op_ADC_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_ADx_r_ptr (get_r1, put_r1) ptr true
	and
	    op_ADD_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_ADx_r_ptr (get_r1, put_r1) ptr false
	and
	    op_ADD_rr_rr
		(get_r1 : cpu_state -> Register16.reg16, put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register16.reg16)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register16.add (Bit.zero, get_r1 cs, get_r2 cs);
					val fl = (resetFlag_N) o (conv16C f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (pad_ticks 4 (pad_ticks 3 f))
		end
	and
	    op_ADC_rr_rr
		(get_r1 : cpu_state -> Register16.reg16, put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register16.reg16)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register16.add (getFlag_C cs, get_r1 cs, get_r2 cs);
					val fl = (resetFlag_N) o (conv16V f)  o (conv16Z f) o (conv16S f) o (conv16C f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (pad_ticks 4 (pad_ticks 3 f))
		end
	and
	    op_SBC_rr_rr
		(get_r1 : cpu_state -> Register16.reg16, put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register16.reg16)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register16.subtract (getFlag_C cs, get_r1 cs, get_r2 cs);
					val fl = (setFlag_N) o (conv16V f)  o (conv16Z f) o (conv16S f) o (conv16C f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (pad_ticks 4 (pad_ticks 3 f))
		end
	and
	    op_SBx_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
		(carry : bool)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if carry then getFlag_C cs else Bit.zero;
					val (r, f) = Register8.subtract (cc, get_r1 cs, get_r2 cs);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_SBC_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_SBx_r_r (get_r1, put_r1) get_r2 true
	and
	    op_SUB_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_SBx_r_r (get_r1, put_r1) get_r2 false
	and
	    op_SBx_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
		(carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if carry then getFlag_C cs else Bit.zero;
					val (r, f) = Register8.subtract (cc, get_r1 cs, ptrval);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oMR 3 f ptr)
		end
	and
	    op_SBC_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_SBx_r_ptr (get_r1, put_r1) ptr true
	and
	    op_SUB_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_SBx_r_ptr (get_r1, put_r1) ptr false
	and
	    op_CP_r_n
		(get_r1 : cpu_state -> Register8.reg8)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.subtract (Bit.zero, get_r1 cs, ptrval);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = fl cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_CP_r_r
		(get_r1 : cpu_state -> Register8.reg8)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.subtract (Bit.zero, get_r1 cs, get_r2 cs);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = fl cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_CP_r_ptr
		(get_r1 : cpu_state -> Register8.reg8)
		(ptr : Register16.reg16)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.subtract (Bit.zero, get_r1 cs, ptrval);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f) o (convC f);
					val ncs = fl cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oMR 3 f ptr)
		end
	and
	    op_logical_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
		(logop)
		(set_half_carry : bool)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if set_half_carry then setFlag_H else resetFlag_H;
					val (r, f) = logop (get_r1 cs, get_r2 cs);
					val fl = (resetFlag_C) o (resetFlag_N) o (convP f) o cc o (convZ f) o (convS f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_logical_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
		(logop)
		(set_half_carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if set_half_carry then setFlag_H else resetFlag_H;
					val (r, f) = logop (get_r1 cs, ptrval);
					val fl = (resetFlag_C) o (resetFlag_N) o (convP f) o cc o (convZ f) o (convS f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oMR 3 f ptr)
		end
	and
	    op_logical_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(logop)
		(set_half_carry : bool)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = if set_half_carry then setFlag_H else resetFlag_H;
					val (r, f) = logop (get_r1 cs, ptrval);
					val fl = (resetFlag_C) o (resetFlag_N) o (convP f) o cc o (convZ f) o (convS f);
					val ncs = ((put_r1 r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
			;
		in
			pad_ticks 0 (oODR 3 f)
		end
	and
	    op_AND_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_logical_r_n (get_r1, put_r1) Register8.andBits true
	and
	    op_OR_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_logical_r_n (get_r1, put_r1) Register8.orBits false
	and
	    op_XOR_r_n
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
	=
		op_logical_r_n (get_r1, put_r1) Register8.xorBits false
	and
	    op_AND_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_logical_r_r (get_r1, put_r1) get_r2 Register8.andBits true
	and
	    op_OR_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_logical_r_r (get_r1, put_r1) get_r2 Register8.orBits false
	and
	    op_XOR_r_r
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register8.reg8)
	=
		op_logical_r_r (get_r1, put_r1) get_r2 Register8.xorBits false
	and
	    op_AND_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_logical_r_ptr (get_r1, put_r1) ptr Register8.andBits true
	and
	    op_OR_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_logical_r_ptr (get_r1, put_r1) ptr Register8.orBits false
	and
	    op_XOR_r_ptr
		(get_r1 : cpu_state -> Register8.reg8, put_r1 : Register8.reg8 -> cpu_state -> cpu_state)
		(ptr : Register16.reg16)
	=
		op_logical_r_ptr (get_r1, put_r1) ptr Register8.xorBits false
	and
	    op_INC_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.add (Bit.zero, get_r cs, Register8.fromInt 1);
					val fl = (resetFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f);
					val ncs = ((put_r r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_INC_ptr
		(adr : Register16.reg16)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.add (Bit.zero, ptrval, Register8.fromInt 1);
					val fl = (resetFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f);
					val ncs =  fl cs;
				in
					pad_ticks 0 (oMW 3 next_instruction adr r) (ncs, bs) 
				end
		in
			pad_ticks 0 (oMR 4 f adr)
		end
	and
	    op_DEC_ptr
		(adr : Register16.reg16)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.subtract (Bit.zero, ptrval, Register8.fromInt 1);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f);
					val ncs =  fl cs;
				in
					pad_ticks 0 (oMW 3 next_instruction adr r) (ncs, bs) 
				end
		in
			pad_ticks 0 (oMR 4 f adr)
		end
	and
	    op_DEC_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.subtract (Bit.zero, get_r cs, Register8.fromInt 1);
					val fl = (setFlag_N) o (convV f) o (convH f) o (convZ f) o (convS f);
					val ncs = ((put_r r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_INC_rr
		(get_r : cpu_state -> Register16.reg16, put_r : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) =
				let
					val (r, f) = Register16.add (Bit.zero, get_r cs, Register16.fromInt 1);
					val ncs = (put_r r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 2 f
		end
	and
	    op_DEC_rr
		(get_r : cpu_state -> Register16.reg16, put_r : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) =
				let
					val (r, f) = Register16.subtract (Bit.zero, get_r cs, Register16.fromInt 1);
					val ncs = (put_r r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 2 f
		end
	and
	    op_RLCA
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlcBits (#A cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H);
					val ncs = ((putA r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RRCA
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrcBits (#A cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H);
					val ncs = ((putA r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RLA
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlBits (getFlag_C cs) (#A cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H);
					val ncs = ((putA r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RRA
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrBits (getFlag_C cs) (#A cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H);
					val ncs = ((putA r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RLC_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlcBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RLC_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlcBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_RRC_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrcBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RRC_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrcBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_SLA_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.slaBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SLA_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.slaBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_SRA_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.sraBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SRA_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.sraBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_SLL_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.sllBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SLL_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.sllBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_SRL_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.srlBits (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SRL_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.srlBits ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (ncs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_RES_r
		n
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = Register8.resetBit n (get_r cs);
					val ncs = (put_r r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RES_ptr
		n
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = Register8.resetBit n ptrval;
				in
					oMW 3 next_instruction adr r (cs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_SET_r
		n
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = Register8.setBit n (get_r cs);
					val ncs = (put_r r) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SET_ptr
		n
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val r = Register8.setBit n ptrval;
				in
					oMW 3 next_instruction adr r (cs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_BIT_r
		n
		(get_r : cpu_state -> Register8.reg8)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val c = Register8.getBit n (get_r cs);
					val cc = if Bit.isSet c then setFlag_Z else resetFlag_Z;
					val fl = cc o (resetFlag_N) o (setFlag_H);
					val ncs = fl cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_BIT_ptr
		n
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val c = Register8.getBit n ptrval;
					val cc = if Bit.isSet c then setFlag_Z else resetFlag_Z;
					val fl = cc o (resetFlag_N) o (setFlag_H);
					val ncs = fl cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_RL_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlBits (getFlag_C cs) (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RL_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rlBits (getFlag_C cs) ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (cs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_RR_r
		(get_r : cpu_state -> Register8.reg8, put_r : Register8.reg8 -> cpu_state -> cpu_state)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrBits (getFlag_C cs) (get_r cs);
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = ((put_r r)o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_RR_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f ptrval (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f, c) = Register8.rrBits (getFlag_C cs) ptrval;
					val cc = if Bit.isSet c then setFlag_C else resetFlag_C;
					val fl = cc o (resetFlag_N) o (resetFlag_H) o (convP f) o (convZ f) o (convS f);
					val ncs = fl cs;
				in
					oMW 3 next_instruction adr r (cs, bs)
				end
		in
			pad_ticks 0 (oMR 4 f adr) (cs, bs)
		end
	and
	    op_CPL
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val fl = (setFlag_N o setFlag_H);
					val ncs = ((putA (#1 (Register8.negateBits (#A cs)))) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_NEG
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val (r, f) = Register8.u2Bits (#A cs);
					val fl = (setFlag_N o (convC f) o (convV f) o (convH f) o (convZ f) o (convS f));
					val ncs = ((putA r) o fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_CCF
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val cc = fn (c : cpu_state) => if Bit.isSet(getFlag_C c) then resetFlag_C c else setFlag_C c;
					val fl = (resetFlag_N o cc);
					val ncs = (fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_SCF
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val fl = (resetFlag_N o setFlag_C);
					val ncs = (fl) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_EX_AF
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = ((putA (#A' cs)) o (putF (#F' cs)) o (putA' (#A cs)) o (putF' (#F cs))) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_EXX
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs =
				{
						A  = #A cs,
						A' = #A' cs,
						F  = #F cs,
						F' = #F' cs,
						B  = #B cs,
						B' = #B' cs,
						C  = #C' cs,
						C' = #C cs,
						D  = #D' cs,
						D' = #D cs,
						E  = #E' cs,
						E' = #E cs,
						H  = #H' cs,
						H' = #H cs,
						L  = #L' cs,
						L' = #L cs,
						I  = #I cs,
						R  = #R cs,
						IX = #IX cs,
						IY = #IY cs,
						SP = #SP cs,
						PC = #PC cs,
						IM = #IM cs,
						IFF1 = #IFF1 cs,
						HLT = #HLT cs,
						IgnoreNextInterrupt = #IgnoreNextInterrupt cs,
						IFF2 = #IFF2 cs,
						inNMI = #inNMI cs,
						HLas = #HLas cs 
					};
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f (cs, bs)
		end
	and
	    op_EX_rr_rr
		(get_r1 : cpu_state -> Register16.reg16, put_r1 : Register16.reg16 -> cpu_state -> cpu_state)
		(get_r2 : cpu_state -> Register16.reg16, put_r2 : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = ((put_r1 (get_r2 cs)) o (put_r2 (get_r1 cs))) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 f
		end
	and
	    op_EX_SP_rr
		(get_r : cpu_state -> Register16.reg16, put_r : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f data (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = put_r data cs;
				in
					oSW16 8 next_instruction (get_r cs)  (ncs,bs)
				end
		in
			pad_ticks 0 (oSR16 (*7*) f)
		end
	and
	    op_DJNZ
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = decB cs;
					fun g (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val t = (decPC o decPC) cs;
							val disp = Register16.fromInt (Register8.asSigned r);
							val (npc, _) = Register16.add (Bit.zero, #PC t, disp);
							val nncs = putPC npc cs;
						in
							(Bus.no_bus_change, DS(nncs, next_instruction))
						end
				in
					if Register8.asUnsigned (#B ncs) <> 0 then
						pad_ticks 5 g (ncs, bs)
					else
						(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 1 (oODR 3 f) (cs, bs)
		end
	and
	    op_JR_cond
		cond
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					fun g (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val t = (decPC o decPC) cs;
							val disp = Register16.fromInt (Register8.asSigned r);
							val (npc, _) = Register16.add (Bit.zero, #PC t, disp);
							val nncs = putPC npc cs;
						in
							(Bus.no_bus_change, DS(nncs, next_instruction))
						end
				in
					if cond cs then
						pad_ticks 5 g (cs, bs)
					else
						(Bus.no_bus_change, DS(cs, next_instruction))
				end
		in
			pad_ticks 0 (oODR 3 f) (cs, bs)
		end
	and
	    op_JR
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f r (cs : cpu_state, bs : Bus.bus_state) = 
				let
					fun g (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val t = (decPC o decPC) cs;
							val disp = Register16.fromInt (Register8.asSigned r);
							val (npc, _) = Register16.add (Bit.zero, #PC t, disp);
							val nncs = putPC npc cs;
						in
							(Bus.no_bus_change, DS(nncs, next_instruction))
						end
				in
					pad_ticks 5 g (cs, bs)
				end
		in
			pad_ticks 0 (oODR 3 f) (cs, bs)
		end
	and
	    op_JP_cond
		cond
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f adr (cs : cpu_state, bs : Bus.bus_state) = 
				let
					fun g (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val nncs = putPC adr cs;
						in
							(Bus.no_bus_change, DS(nncs, next_instruction))
						end
				in
					if cond cs then
						pad_ticks 0 g (cs, bs)
					else
						(Bus.no_bus_change, DS(cs, next_instruction))
				end
		in
			pad_ticks 0 (oODR16 f) (cs, bs)
		end
	and
	    op_JP
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f adr (cs : cpu_state, bs : Bus.bus_state) = 
				let
					fun g (cs : cpu_state, bs : Bus.bus_state) = 
						let
							val nncs = putPC adr cs;
						in
							(Bus.no_bus_change, DS(nncs, next_instruction))
						end
				in
					pad_ticks 0 g (cs, bs)
				end
		in
			pad_ticks 0 (oODR16 f) (cs, bs)
		end
	and
	    op_JP_ptr
		adr
		(cs : cpu_state, bs : Bus.bus_state)
	=
		pad_ticks 0 next_instruction (putPC adr cs, bs)
	and
	    op_RST_adr
		(adr)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f (cs : cpu_state, bs : Bus.bus_state) =
				(Bus.no_bus_change, DS(putPC (Register16.fromInt adr) cs, next_instruction))
			;
		in
			pad_ticks 1 (oSW16 6 f (#PC cs)) (cs, bs)
		end
	and
	    op_CALL_cond
		cond
	=
		let
			fun g adr (cs : cpu_state, bs : Bus.bus_state) = 
			let
				fun f (cs : cpu_state, bs : Bus.bus_state) =
					(Bus.no_bus_change, DS(putPC adr cs, next_instruction))
				;
			in
				if cond cs then
					oSW16 6 f (#PC cs) (cs, bs)
				else
					(Bus.no_bus_change, DS(cs, next_instruction))
			end
		in
			oODR16 g
		end
	and
	    op_CALL
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun g adr (cs : cpu_state, bs : Bus.bus_state) = 
			let
				fun f (cs : cpu_state, bs : Bus.bus_state) =
					(Bus.no_bus_change, DS(putPC adr cs, next_instruction))
				;
			in
				oSW16 6 f (#PC cs) (cs, bs)
			end
		in
			oODR16 g (cs, bs)
		end
	and
	    op_RET
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let
			fun f adr (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = (putPC adr) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oSR16 f) (cs, bs)
		end
	and
	    op_RET_cond
		cond
		(cs : cpu_state, bs : Bus.bus_state)
	=
		if (cond cs) then
			op_RET (cs, bs)
		else
			pad_ticks 0 next_instruction (cs, bs)
	and
	    op_POP_rr
		(put_r : Register16.reg16 -> cpu_state -> cpu_state)
	=
		let
			fun f data (cs : cpu_state, bs : Bus.bus_state) = 
				let
					val ncs = (put_r data) cs;
				in
					(Bus.no_bus_change, DS(ncs, next_instruction))
				end
		in
			pad_ticks 0 (oSR16 f)
		end
	and
	    op_PUSH_rr
		(get_r : cpu_state -> Register16.reg16)
		(cs : cpu_state, bs : Bus.bus_state)
	=
		pad_ticks 1 (oSW16 6 next_instruction (get_r cs)) (cs, bs)
	and
	   HL2IX 
		cont 
		(cs : cpu_state, bs : Bus.bus_state)
	=
		let 
			fun adddisp disp = 
				let
					val i = Register8.asSigned disp;
			(*?*)		val adr = Register16.fromInt ((Register16.asUnsigned (rplGetHL cs cs)) + i);
				in
					pad_ticks 5 (cont adr)
				end
		in
			if #HLas cs = asHL then
				cont (getHL cs) (cs, bs)
			else
				oODR 3 adddisp (cs, bs)
		end
	and
	    choose_opcode_CB
		(opcode : Register8.reg8)
		(s : cpu_state * Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		let 
			val cs = #1 s;
			val bs = #2 s;
			val opc = Register8.asUnsigned opcode;
			val opbf = Register8.asBitfield opcode;
			val reg = Bitfield.asUnsigned (Bitfield.subField 0 3 opbf);
			val rest = Bitfield.asUnsigned (Bitfield.subField 3 3 opbf);
			val hl = if reg = 6 then true else false;
			fun regop 0 cs
			  : ((cpu_state -> Register8.reg8) * (Register8.reg8 -> cpu_state -> cpu_state))
 
				    = (#B, putB)
			  | regop 1 cs = (#C, putC)
			  | regop 2 cs = (#D, putD)
			  | regop 3 cs = (#E, putE)
			  | regop 4 cs = (rplGetH cs, rplPutH cs)
			  | regop 5 cs = (rplGetL cs, rplPutL cs)
			  | regop 7 cs = (#A, putA)
			  | regop _ _= raise Fail ("opcode CD: regop")
			;
			fun reset i = (pr ("RES" ^ intPadded 3 i ^ "\n"); 
					if hl then 
					op_RES_ptr i (rplGetHL cs cs) (cs, bs)
				else 
					op_RES_r i (regop reg cs) (cs, bs)
				);
			fun set i = (pr ("SET" ^ intPadded 3 i ^ "\n"); 
					if hl then 
					op_SET_ptr i (rplGetHL cs cs) (cs, bs)
				else 
					op_SET_r i (regop reg cs) (cs, bs)
				);
			fun bit i = (pr ("BIT" ^ intPadded 3 i ^ "\n"); 
					if hl then 
					op_BIT_ptr i (rplGetHL cs cs) (cs, bs)
				else 
					op_BIT_r i (#1 (regop reg cs)) (cs, bs)
				);

		in
			case rest of
			  0  => (pr "RLC\n" ; if hl then 
					op_RLC_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_RLC_r (regop reg cs) (cs, bs)
				)
			| 1  => (pr "RRC\n" ; if hl then 
					op_RRC_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_RRC_r (regop reg cs) (cs, bs)
				)
			| 2  => (pr "RL\n" ; if hl then 
					op_RL_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_RL_r (regop reg cs) (cs, bs)
				)
			| 3  => (pr "RR\n" ; if hl then 
					op_RR_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_RR_r (regop reg cs) (cs, bs)
				)
			| 4  => (pr "SLA\n" ; if hl then 
					op_SLA_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_SLA_r (regop reg cs) (cs, bs)
				)
			| 5  => (pr "SRA\n" ; if hl then 
					op_SRA_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_SRA_r (regop reg cs) (cs, bs)
				)
			| 6  => (pr "SLL\n" ; if hl then 
					op_SLL_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_SLL_r (regop reg cs) (cs, bs)
				)
			| 7  => (pr "SRL\n" ; if hl then 
					op_SRL_ptr (rplGetHL cs cs) (cs, bs)
				else 
					op_SRL_r (regop reg cs) (cs, bs)
				)
			| 8 => bit 0
			| 9 => bit 1
			| 10 => bit 2
			| 11 => bit 3
			| 12 => bit 4
			| 13 => bit 5
			| 14 => bit 6
			| 15 => bit 7
			| 16 => reset 0
			| 17 => reset 1
			| 18 => reset 2
			| 19 => reset 3
			| 20 => reset 4
			| 21 => reset 5
			| 22 => reset 6
			| 23 => reset 7
			| 24 => set 0
			| 25 => set 1
			| 26 => set 2
			| 27 => set 3
			| 28 => set 4
			| 29 => set 5
			| 30 => set 6
			| 31 => set 7
			| z   => (pr ("prefix CB:"^unknown_opcode_string z cs) ; 
				raise UNKNOWN_OPCODE {opcode = opcode, PC = (#PC cs)})
		end
	and
	    choose_opcode_ED
		(opcode : Register8.reg8)
		(s : cpu_state * Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		let 
			val cs = #1 s;
			val bs = #2 s;
		in
(* DD and FD prefixes are not working with ED prefix *)
			pr ((cpuAsString cs) ^ "\n"); 
			case Register8.asUnsigned opcode of
(* IN B,(C)   *)	  64   => (pr "IN B, (C)\n" ; op_IN_r_C putB (cs, bs))
(* OUT (C),B  *)	| 65   => (pr "OUT (C),B\n" ; op_OUT_C_r (#B) (cs, bs))
(* SBC HL, BC *)	| 66   => (pr "SBC HL, BC\n" ; op_SBC_rr_rr (getHL, putHL) getBC (cs, bs))
(* LD (nn),BC *)	| 67   => (pr "LD (nn), BC\n" ; op_LD_ptrnn_rr getBC (cs, bs))
(* NEG        *)	| 68   => (pr "NEG\n" ; op_NEG (cs, bs))
(* RETN       *)	| 69   => (pr "RETN\n" ; op_RET ((resetInNMI o (putIFF1 (#IFF2 cs))) cs, bs))
(* IM 0       *)	| 70   => (pr "IM 0\n" ; op_IM 0 (cs, bs))
(* LD I,A     *)	| 71   => (pr "LD I,A\n" ; op_LD_r_r putI (#A) (cs, bs))
(* IN C,(C)   *)	| 72   => (pr "IN C, (C)\n" ; op_IN_r_C putC (cs, bs))
(* OUT (C),C  *)	| 73   => (pr "OUT (C),C\n" ; op_OUT_C_r (#C) (cs, bs))
(* ADC HL, BC *)	| 74   => (pr "ADC HL, BC\n" ; op_ADC_rr_rr (getHL, putHL) getBC (cs, bs))
(* LD BC,(nn) *)	| 75   => (pr "LD BC, (nn)\n" ; op_LD_rr_ptrnn putBC (cs, bs))
(* RETI       *)	| 77   => (pr "RETI\n" ; op_RET ((resetInNMI o (putIFF1 (#IFF2 cs))) cs, bs))
(* LD R,A     *)	| 79   => (pr "LD R,A\n" ; op_LD_r_r putR (#A) (cs, bs))
(* IN D,(C)   *)	| 80   => (pr "IN D, (C)\n" ; op_IN_r_C putD (cs, bs))
(* OUT (C),D  *)	| 81   => (pr "OUT (C),D\n" ; op_OUT_C_r (#D) (cs, bs))
(* SBC HL, DE *)	| 82   => (pr "SBC HL, DE\n" ; op_SBC_rr_rr (getHL, putHL) getDE (cs, bs))
(* LD (nn),DE *)	| 83   => (pr "LD (nn), DE\n" ; op_LD_ptrnn_rr getDE (cs, bs))
(* IM 1       *)	| 86   => (pr "IM 1\n" ; op_IM 1 (cs, bs))
(* LD A,I     *)	| 87   => (pr "LD A,I\n" ; op_LD_r_specr putA (#I) (cs, bs))
(* IN E,(C)   *)	| 88   => (pr "IN E, (C)\n" ; op_IN_r_C putE (cs, bs))
(* OUT (C),E  *)	| 89   => (pr "OUT (C),E\n" ; op_OUT_C_r (#E) (cs, bs))
(* ADC HL, DE *)	| 90   => (pr "ADC HL, DE\n" ; op_ADC_rr_rr (getHL, putHL) getDE (cs, bs))
(* LD DE,(nn) *)	| 91   => (pr "LD DE, (nn)\n" ; op_LD_rr_ptrnn putDE (cs, bs))
(* IM 2       *)	| 94   => (pr "IM 2\n" ; op_IM 2 (cs, bs))
(* LD A,R     *)	| 95   => (pr "LD A,R\n" ; op_LD_r_specr putA (#R) (cs, bs))
(* IN H,(C)   *)	| 96   => (pr "IN H, (C)\n" ; op_IN_r_C putH (cs, bs))
(* OUT (C),H  *)	| 97   => (pr "OUT (C),H\n" ; op_OUT_C_r (#H) (cs, bs))
(* SBC HL, HL *)	| 98   => (pr "SBC HL, HL\n" ; op_SBC_rr_rr (getHL, putHL) (getHL) (cs, bs))
(* LD (nn),HL *)	| 99   => (pr "LD (nn), HL\n" ; op_LD_ptrnn_rr getHL (cs, bs))
(* IN L,(C)   *)	| 104  => (pr "IN L, (C)\n" ; op_IN_r_C putL (cs, bs))
(* OUT (C),L  *)	| 105  => (pr "OUT (C),L\n" ; op_OUT_C_r (#L) (cs, bs))
(* ADC HL, HL *)	| 106   => (pr "ADC HL, HL\n" ; op_ADC_rr_rr (getHL, putHL) (getHL) (cs, bs))
(* LD HL,(nn) *)	| 107   => (pr "LD HL, (nn)\n" ; op_LD_rr_ptrnn (putHL) (cs, bs))
(* IN (C)     *)	| 112  => (pr "IN (C)\n" ; op_IN_r_C (fn _ => fn c => c) (cs, bs))
(* OUT (C),0  *)	| 113  => (pr "OUT (C),A\n" ; op_OUT_C_r (fn c => Register8.zero) (cs, bs))
(* SBC HL, SP *)	| 114  => (pr "SBC HL, SP\n" ; op_SBC_rr_rr (getHL, putHL) (#SP) (cs, bs))
(* LD (nn),SP *)	| 115   => (pr "LD (nn), SP\n" ; op_LD_ptrnn_rr (#SP) (cs, bs))
(* IN A,(C)   *)	| 120  => (pr "IN A, (C)\n" ; op_IN_r_C putA (cs, bs))
(* OUT (C),A  *)	| 121  => (pr "OUT (C),A\n" ; op_OUT_C_r (#A) (cs, bs))
(* ADC HL, SP *)	| 122   => (pr "ADC HL, SP\n" ; op_ADC_rr_rr (getHL, putHL) (#SP) (cs, bs))
(* LD SP,(nn) *)	| 123   => (pr "LD SP, (nn)\n" ; op_LD_rr_ptrnn (putSP) (cs, bs))
(* LDI        *)	| 160   => (pr "LDI\n" ; op_LDxx true false (cs, bs))
(* CPI        *)	| 161   => (pr "CPI\n" ; op_CPxx true false (cs, bs))
(* LDD        *)	| 168   => (pr "LDD\n" ; op_LDxx false false (cs, bs))
(* CPD        *)	| 169   => (pr "CPD\n" ; op_CPxx false false (cs, bs))
(* LDIR       *)	| 176   => (pr "LDIR\n" ; op_LDxx true true (cs, bs))
(* CPIR       *)	| 177   => (pr "CPIR\n" ; op_CPxx true true (cs, bs))
(* LDDR        *)	| 184   => (pr "LDDR\n" ; op_LDxx false true (cs, bs))
(* CPDR        *)	| 185   => (pr "CPDR\n" ; op_CPxx false true (cs, bs))
			| z    => (pr ("Prefix ED:" ^ (unknown_opcode_string z cs)); 
				raise UNKNOWN_OPCODE {opcode = opcode, PC = (#PC cs)})
		end		
	and
	    choose_opcode
		(opcode : Register8.reg8)
		(s : cpu_state * Bus.bus_state)
	:
		Bus.bus_change * device_state
	=
		let 
			val cs = #1 s;
			val bs = #2 s;
		in
			pr ((cpuAsString cs) ^ "\n"); 
			case Register8.asUnsigned opcode of
(* NOP      *)		  0   => (pr "NOP\n" ; op_NOP (cs, bs))
(* LD BC,nn *)		| 1   => (pr "LD BC,nn\n" ; op_LD_r_nn putBC (cs, bs)) 
(* LD (BC),A *)		| 2   => (pr "LD (BC),A\n" ; op_LD_ptr_r (#A) (getBC cs) (cs, bs)) 
(* INC BC   *)		| 3   => (pr "INC BC\n" ; op_INC_rr (getBC, putBC) (cs, bs)) 
(* INC B    *)		| 4   => (pr "INC B\n" ; op_INC_r (#B, putB) (cs, bs)) 
(* DEC B    *)		| 5   => (pr "DEC B\n" ; op_DEC_r (#B, putB) (cs, bs)) 
(* LD B,n   *)		| 6   => (pr "LD B,n\n" ; op_LD_r_n putB (cs, bs)) 
(* RLCA     *)		| 7   => (pr "RLCA\n" ; op_RLCA (cs, bs)) 
(* EX AF, AF'*)		| 8   => (pr "EX AF,AF'\n" ; op_EX_AF (cs, bs)) 
(* ADD HL,BC *)		| 9   => (pr "ADD HL,BC\n" ; op_ADD_rr_rr (rplGetHL cs, rplPutHL cs) (getBC) (cs, bs)) 
(* LD A,(BC)*)		| 10  => (pr "LD A,(BC)\n" ; op_LD_r_ptr putA (getBC cs) (cs, bs)) 
(* DEC BC   *)		| 11   => (pr "DEC BC\n" ; op_DEC_rr (getBC, putBC) (cs, bs)) 
(* INC C    *)		| 12  => (pr "INC C\n" ; op_INC_r (#C, putC) (cs, bs)) 
(* DEC C    *)		| 13  => (pr "DEC C\n" ; op_DEC_r (#C, putC) (cs, bs)) 
(* LD C,n   *)		| 14  => (pr "LD C,n\n" ; op_LD_r_n putC (cs, bs)) 
(* RRCA     *)		| 15  => (pr "RRCA\n" ; op_RRCA (cs, bs)) 
(* DJNZ e   *)		| 16  => (pr "DJNZ e\n" ; op_DJNZ (cs, bs)) 
(* LD DE,nn *)		| 17  => (pr "LD DE,nn\n" ; op_LD_r_nn putDE (cs, bs)) 
(* LD (DE),A *)		| 18   => (pr "LD (DE),A\n" ; op_LD_ptr_r (#A) (getDE cs) (cs, bs))
(* INC DE   *)		| 19  => (pr "INC DE\n" ; op_INC_rr (getDE, putDE) (cs, bs)) 
(* INC D    *)		| 20  => (pr "INC D\n" ; op_INC_r (#D, putD) (cs, bs)) 
(* DEC D    *)		| 21  => (pr "DEC D\n" ; op_DEC_r (#D, putD) (cs, bs)) 
(* LD D,n   *)		| 22  => (pr "LD D,n\n" ; op_LD_r_n putD (cs, bs)) 
(* RLA      *)		| 23  => (pr "RLA\n" ; op_RLA (cs, bs)) 
(* JR e     *)		| 24  => (pr "JR e\n" ; op_JR (cs, bs)) 
(* ADD HL,DE *)		| 25   => (pr "ADD HL,DE\n" ; op_ADD_rr_rr (rplGetHL cs, rplPutHL cs) (getDE) (cs, bs)) 
(* LD A,(DE)*)		| 26  => (pr "LD A,(DE)\n" ; op_LD_r_ptr putA (getDE cs) (cs, bs)) 
(* DEC DE   *)		| 27   => (pr "DEC DE\n" ; op_DEC_rr (getDE, putDE) (cs, bs)) 
(* INC E    *)		| 28  => (pr "INC E\n" ; op_INC_r (#E, putE) (cs, bs))
(* DEC E    *)		| 29  => (pr "DEC E\n" ; op_DEC_r (#E, putE) (cs, bs)) 
(* LD E,n   *)		| 30  => (pr "LD E,n\n" ; op_LD_r_n putE (cs, bs)) 
(* RRA     *)		| 31  => (pr "RRA\n" ; op_RRA (cs, bs)) 
(* JR NZ,e     *)	| 32  => (pr "JR NZ,e\n" ; op_JR_cond (fn cs => Bit.isNotSet (getFlag_Z cs)) (cs, bs)) 
(* LD HL,nn *)		| 33  => (pr "LD HL,nn\n" ; op_LD_r_nn (rplPutHL cs) (cs, bs)) 
(* LD (nn),HL *)	| 34  => (pr "LD (nn),HL\n" ; op_LD_ptrnn_rr (rplGetHL cs) (cs, bs)) 
(* INC HL   *)		| 35  => (pr "INC HL\n" ; op_INC_rr (rplGetHL cs, rplPutHL cs) (cs, bs)) 
(* INC H    *)		| 36  => (pr "INC H\n" ; op_INC_r (rplGetH cs, rplPutH cs) (cs, bs))
(* DEC H    *)		| 37  => (pr "DEC H\n" ; op_DEC_r (rplGetH cs, rplPutH cs) (cs, bs)) 
(* LD H,n   *)		| 38  => (pr "LD H,n\n" ; op_LD_r_n (rplPutH cs) (cs, bs)) 
(* JR Z,e     *)	| 40  => (pr "JR Z,e\n" ; op_JR_cond (fn cs => Bit.isSet (getFlag_Z cs)) (cs, bs)) 
(* ADD HL,HL *)		| 41   => (pr "ADD HL,HL\n" ; op_ADD_rr_rr (rplGetHL cs, rplPutHL cs) (rplGetHL cs) (cs, bs)) 
(* LD HL,(nn) *)	| 42   => (pr "LD HL,(nn)\n" ; op_LD_rr_ptrnn (rplPutHL cs) (cs, bs)) 
(* DEC HL   *)		| 43   => (pr "DEC HL\n" ; op_DEC_rr (rplGetHL cs, rplPutHL cs) (cs, bs)) 
(* INC L    *)		| 44  => (pr "INC L\n" ; op_INC_r (rplGetL cs, rplPutL cs) (cs, bs))
(* DEC L    *)		| 45  => (pr "DEC L\n" ; op_DEC_r (rplGetL cs, rplPutL cs) (cs, bs)) 
(* LD L,n   *)		| 46  => (pr "LD L,n\n" ; op_LD_r_n (rplPutL cs) (cs, bs)) 
(* CPL   *)		| 47  => (pr "CPL\n" ; op_CPL (cs, bs)) 
(* JR NC,e     *)	| 48  => (pr "JR NC,e\n" ; op_JR_cond (fn cs => Bit.isNotSet (getFlag_C cs)) (cs, bs)) 
(* LD SP,nn *)		| 49  => (pr "LD SP,nn\n" ; op_LD_r_nn putSP (cs, bs)) 
(* LD (nn),A   *)	| 50  => (pr "LD (nn),A\n" ; op_LD_ptrnn_r (#A) (cs, bs)) 
(* INC SP   *)		| 51  => (pr "INC SP\n" ; op_INC_rr (#SP, putSP) (cs, bs)) 
(* INC (HL)   *)	| 52  => (pr "INC (HL)\n" ; HL2IX (op_INC_ptr) (cs, bs)) 
(* DEC (HL)   *)	| 53  => (pr "DEC (HL)\n" ; HL2IX (op_DEC_ptr) (cs, bs)) 
(* LD (HL),n   *)	| 54  => (pr "LD (HL),n\n" ; HL2IX op_LD_ptr_n (cs, bs)) 
(* SCF   *)		| 55  => (pr "SCF\n" ; op_SCF (cs, bs)) 
(* JR C,e     *)	| 56  => (pr "JR C,e\n" ; op_JR_cond (fn cs => Bit.isSet (getFlag_C cs)) (cs, bs)) 
(* ADD HL,SP *)		| 57   => (pr "ADD HL,SP\n" ; op_ADD_rr_rr (rplGetHL cs, rplPutHL cs) (#SP) (cs, bs)) 
(* LD A,(nn) *)		| 58   => (pr "LD A,(nn)\n" ; op_LD_r_ptrnn putA (cs, bs)) 
(* DEC SP   *)		| 59   => (pr "DEC SP\n" ; op_DEC_rr (#SP, putSP) (cs, bs)) 
(* INC A    *)		| 60  => (pr "INC A\n" ; op_INC_r (#A, putA) (cs, bs)) 
(* DEC A    *)		| 61  => (pr "DEC A\n" ; op_DEC_r (#A, putA) (cs, bs)) 
(* LD A,n   *)		| 62  => (pr "LD A,n\n" ; op_LD_r_n putA (cs, bs)) 
(* CCF   *)		| 63  => (pr "CCF\n" ; op_CCF (cs, bs)) 
(* LD B,B   *)		| 64  => (pr "LD B,B\n" ; op_LD_r_r putB (#B) (cs, bs)) 
(* LD B,C   *)		| 65  => (pr "LD B,C\n" ; op_LD_r_r putB (#C) (cs, bs)) 
(* LD B,D   *)		| 66  => (pr "LD B,D\n" ; op_LD_r_r putB (#D) (cs, bs)) 
(* LD B,E   *)		| 67  => (pr "LD B,E\n" ; op_LD_r_r putB (#E) (cs, bs)) 
(* LD B,H   *)		| 68  => (pr "LD B,H\n" ; op_LD_r_r putB (rplGetH cs) (cs, bs)) 
(* LD B,L   *)		| 69  => (pr "LD B,L\n" ; op_LD_r_r putB (rplGetL cs) (cs, bs)) 
(* LD B,(HL)*)		| 70  => (pr "LD B,(HL)\n" ; HL2IX (op_LD_r_ptr putB) (cs, bs)) 
(* LD B,A   *)		| 71  => (pr "LD B,A\n" ; op_LD_r_r putB (#A) (cs, bs)) 
(* LD C,B   *)		| 72  => (pr "LD C,B\n" ; op_LD_r_r putC (#B) (cs, bs)) 
(* LD C,C   *)		| 73  => (pr "LD C,C\n" ; op_LD_r_r putC (#C) (cs, bs)) 
(* LD C,D   *)		| 74  => (pr "LD C,D\n" ; op_LD_r_r putC (#D) (cs, bs)) 
(* LD C,E   *)		| 75  => (pr "LD C,E\n" ; op_LD_r_r putC (#E) (cs, bs)) 
(* LD C,H   *)		| 76  => (pr "LD C,H\n" ; op_LD_r_r putC (rplGetH cs) (cs, bs)) 
(* LD C,L   *)		| 77  => (pr "LD C,L\n" ; op_LD_r_r putC (rplGetL cs) (cs, bs)) 
(* LD C,(HL)*)		| 78  => (pr "LD C,(HL)\n" ; HL2IX (op_LD_r_ptr putC) (cs, bs))
(* LD C,A   *)		| 79  => (pr "LD C,A\n" ; op_LD_r_r putC (#A) (cs, bs)) 
(* LD D,B   *)		| 80  => (pr "LD D,B\n" ; op_LD_r_r putD (#B) (cs, bs)) 
(* LD D,C   *)		| 81  => (pr "LD D,C\n" ; op_LD_r_r putD (#C) (cs, bs)) 
(* LD D,D   *)		| 82  => (pr "LD D,D\n" ; op_LD_r_r putD (#D) (cs, bs)) 
(* LD D,E   *)		| 83  => (pr "LD D,E\n" ; op_LD_r_r putD (#E) (cs, bs)) 
(* LD D,H   *)		| 84  => (pr "LD D,H\n" ; op_LD_r_r putD (rplGetH cs) (cs, bs)) 
(* LD D,L   *)		| 85  => (pr "LD D,L\n" ; op_LD_r_r putD (rplGetL cs) (cs, bs)) 
(* LD D,(HL)*)		| 86  => (pr "LD D,(HL)\n" ; HL2IX (op_LD_r_ptr putD) (cs, bs)) 
(* LD D,A   *)		| 87  => (pr "LD D,A\n" ; op_LD_r_r putD (#A) (cs, bs)) 
(* LD E,B   *)		| 88  => (pr "LD E,B\n" ; op_LD_r_r putE (#B) (cs, bs)) 
(* LD E,C   *)		| 89  => (pr "LD E,C\n" ; op_LD_r_r putE (#C) (cs, bs)) 
(* LD E,D   *)		| 90  => (pr "LD E,D\n" ; op_LD_r_r putE (#D) (cs, bs)) 
(* LD E,E   *)		| 91  => (pr "LD E,E\n" ; op_LD_r_r putE (#E) (cs, bs)) 
(* LD E,H   *)		| 92  => (pr "LD E,H\n" ; op_LD_r_r putE (rplGetH cs) (cs, bs)) 
(* LD E,L   *)		| 93  => (pr "LD E,L\n" ; op_LD_r_r putE (rplGetL cs) (cs, bs)) 
(* LD E,(HL)*)		| 94  => (pr "LD E,(HL)\n" ; HL2IX (op_LD_r_ptr putE) (cs, bs)) 
(* LD E,A   *)		| 95  => (pr "LD E,A\n" ; op_LD_r_r putE (#A) (cs, bs)) 
(* LD H,B   *)		| 96  => (pr "LD H,B\n" ; op_LD_r_r (rplPutH cs) (#B) (cs, bs)) 
(* LD H,C   *)		| 97  => (pr "LD H,C\n" ; op_LD_r_r (rplPutH cs)  (#C) (cs, bs)) 
(* LD H,D   *)		| 98  => (pr "LD H,D\n" ; op_LD_r_r (rplPutH cs)  (#D) (cs, bs)) 
(* LD H,E   *)		| 99  => (pr "LD H,E\n" ; op_LD_r_r (rplPutH cs)  (#E) (cs, bs)) 
(* LD H,H   *)		| 100 => (pr "LD H,H\n" ; op_LD_r_r (rplPutH cs)  (rplGetH cs) (cs, bs)) 
(* LD H,L   *)		| 101 => (pr "LD H,L\n" ; op_LD_r_r (rplPutH cs)  (rplGetL cs) (cs, bs)) 
(* LD H,(HL)*)		| 102 => (pr "LD H,(HL)\n" ; HL2IX (op_LD_r_ptr putH) (cs, bs))  (* YES - here must be putH*)
(* LD H,A   *)		| 103 => (pr "LD H,A\n" ; op_LD_r_r (rplPutH cs)  (#A) (cs, bs)) 
(* LD L,B   *)		| 104 => (pr "LD L,B\n" ; op_LD_r_r (rplPutL cs)  (#B) (cs, bs)) 
(* LD L,C   *)		| 105 => (pr "LD L,C\n" ; op_LD_r_r (rplPutL cs)  (#C) (cs, bs)) 
(* LD L,D   *)		| 106 => (pr "LD L,D\n" ; op_LD_r_r (rplPutL cs)  (#D) (cs, bs)) 
(* LD L,E   *)		| 107 => (pr "LD L,E\n" ; op_LD_r_r (rplPutL cs)  (#E) (cs, bs)) 
(* LD L,H   *)		| 108 => (pr "LD L,H\n" ; op_LD_r_r (rplPutL cs)  (rplGetH cs) (cs, bs)) 
(* LD L,L   *)		| 109 => (pr "LD L,L\n" ; op_LD_r_r (rplPutL cs)  (rplGetL cs) (cs, bs)) 
(* LD L,(HL)*)		| 110 => (pr "LD L,(HL)\n" ; HL2IX (op_LD_r_ptr putL) (cs, bs)) (* Here too - putL *)
(* LD L,A   *)		| 111 => (pr "LD L,A\n" ; op_LD_r_r (rplPutL cs)  (#A) (cs, bs)) 
(* LD (HL),B *)		| 112 => (pr "LD (HL),B\n" ; HL2IX (op_LD_ptr_r (#B)) (cs, bs)) 
(* LD (HL),C *)		| 113 => (pr "LD (HL),C\n" ; HL2IX (op_LD_ptr_r (#C)) (cs, bs)) 
(* LD (HL),D *)		| 114 => (pr "LD (HL),D\n" ; HL2IX (op_LD_ptr_r (#D)) (cs, bs)) 
(* LD (HL),E *)		| 115 => (pr "LD (HL),E\n" ; HL2IX (op_LD_ptr_r (#E)) (cs, bs)) 
(* LD (HL),H *)		| 116 => (pr "LD (HL),H\n" ; HL2IX (op_LD_ptr_r (#H)) (cs, bs)) (* Yes - #H *) 
(* LD (HL),L *)		| 117 => (pr "LD (HL),L\n" ; HL2IX (op_LD_ptr_r (#L)) (cs, bs)) (* #L - the same *) 
(* LD (HL),A *)		| 119 => (pr "LD (HL),A\n" ; HL2IX (op_LD_ptr_r (#A)) (cs, bs)) 
(* LD A,B   *)		| 120 => (pr "LD A,B\n" ; op_LD_r_r putA (#B) (cs, bs)) 
(* LD A,C   *)		| 121 => (pr "LD A,C\n" ; op_LD_r_r putA (#C) (cs, bs)) 
(* LD A,D   *)		| 122 => (pr "LD A,D\n" ; op_LD_r_r putA (#D) (cs, bs)) 
(* LD A,E   *)		| 123 => (pr "LD A,E\n" ; op_LD_r_r putA (#E) (cs, bs)) 
(* LD A,H   *)		| 124 => (pr "LD A,H\n" ; op_LD_r_r putA (rplGetH cs) (cs, bs)) 
(* LD A,L   *)		| 125 => (pr "LD A,L\n" ; op_LD_r_r putA (rplGetL cs) (cs, bs)) 
(* LD A,(HL)*)		| 126 => (pr "LD A,(HL)\n" ; HL2IX (op_LD_r_ptr putA) (cs, bs)) 
(* LD A,A   *)		| 127 => (pr "LD A,A\n" ; op_LD_r_r putA (#A) (cs, bs)) 

(* ADD A, B *)		| 128 => (pr "ADD A,B\n" ; op_ADD_r_r (#A, putA) (#B) (cs, bs)) 
(* ADD A, C *)		| 129 => (pr "ADD A,C\n" ; op_ADD_r_r (#A, putA) (#C) (cs, bs)) 
(* ADD A, D *)		| 130 => (pr "ADD A,D\n" ; op_ADD_r_r (#A, putA) (#D) (cs, bs)) 
(* ADD A, E *)		| 131 => (pr "ADD A,E\n" ; op_ADD_r_r (#A, putA) (#E) (cs, bs)) 
(* ADD A, H *)		| 132 => (pr "ADD A,H\n" ; op_ADD_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* ADD A, L *)		| 133 => (pr "ADD A,L\n" ; op_ADD_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* ADD A, (HL) *)	| 134 => (pr "ADD A,(HL)\n" ; HL2IX (op_ADD_r_ptr (#A, putA)) (cs, bs))
(* ADD A, A *)		| 135 => (pr "ADD A,A\n" ; op_ADD_r_r (#A, putA) (#A) (cs, bs)) 
(* ADC A, B *)		| 136 => (pr "ADC A,B\n" ; op_ADC_r_r (#A, putA) (#B) (cs, bs)) 
(* ADC A, C *)		| 137 => (pr "ADC A,C\n" ; op_ADC_r_r (#A, putA) (#C) (cs, bs)) 
(* ADC A, D *)		| 138 => (pr "ADC A,D\n" ; op_ADC_r_r (#A, putA) (#D) (cs, bs)) 
(* ADC A, E *)		| 139 => (pr "ADC A,E\n" ; op_ADC_r_r (#A, putA) (#E) (cs, bs)) 
(* ADC A, H *)		| 140 => (pr "ADC A,H\n" ; op_ADC_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* ADC A, L *)		| 141 => (pr "ADC A,L\n" ; op_ADC_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* ADC A, (HL) *)	| 142 => (pr "ADC A,(HL)\n" ; HL2IX (op_ADC_r_ptr (#A, putA)) (cs, bs))
(* ADC A, A *)		| 143 => (pr "ADC A,A\n" ; op_ADC_r_r (#A, putA) (#A) (cs, bs)) 
(* SUB A, B *)		| 144 => (pr "SUB A,B\n" ; op_SUB_r_r (#A, putA) (#B) (cs, bs)) 
(* SUB A, C *)		| 145 => (pr "SUB A,C\n" ; op_SUB_r_r (#A, putA) (#C) (cs, bs)) 
(* SUB A, D *)		| 146 => (pr "SUB A,D\n" ; op_SUB_r_r (#A, putA) (#D) (cs, bs)) 
(* SUB A, E *)		| 147 => (pr "SUB A,E\n" ; op_SUB_r_r (#A, putA) (#E) (cs, bs)) 
(* SUB A, H *)		| 148 => (pr "SUB A,H\n" ; op_SUB_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* SUB A, L *)		| 149 => (pr "SUB A,L\n" ; op_SUB_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* SUB A, (HL) *)	| 150 => (pr "SUB A,(HL)\n" ; HL2IX (op_SUB_r_ptr (#A, putA)) (cs, bs))
(* SUB A, A *)		| 151 => (pr "SUB A,A\n" ; op_SUB_r_r (#A, putA) (#A) (cs, bs)) 
(* SBC A, B *)		| 152 => (pr "SBC A,B\n" ; op_SBC_r_r (#A, putA) (#B) (cs, bs)) 
(* SBC A, C *)		| 153 => (pr "SBC A,C\n" ; op_SBC_r_r (#A, putA) (#C) (cs, bs)) 
(* SBC A, D *)		| 154 => (pr "SBC A,D\n" ; op_SBC_r_r (#A, putA) (#D) (cs, bs)) 
(* SBC A, E *)		| 155 => (pr "SBC A,E\n" ; op_SBC_r_r (#A, putA) (#E) (cs, bs)) 
(* SBC A, H *)		| 156 => (pr "SBC A,H\n" ; op_SBC_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* SBC A, L *)		| 157 => (pr "SBC A,L\n" ; op_SBC_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* SBC A, (HL) *)	| 158 => (pr "SBC A,(HL)\n" ; HL2IX (op_SBC_r_ptr (#A, putA)) (cs, bs))
(* SBC A, A *)		| 159 => (pr "SBC A,A\n" ; op_SBC_r_r (#A, putA) (#A) (cs, bs)) 
(* AND A, B *)		| 160 => (pr "AND A,B\n" ; op_AND_r_r (#A, putA) (#B) (cs, bs)) 
(* AND A, C *)		| 161 => (pr "AND A,C\n" ; op_AND_r_r (#A, putA) (#C) (cs, bs)) 
(* AND A, D *)		| 162 => (pr "AND A,D\n" ; op_AND_r_r (#A, putA) (#D) (cs, bs)) 
(* AND A, E *)		| 163 => (pr "AND A,E\n" ; op_AND_r_r (#A, putA) (#E) (cs, bs)) 
(* AND A, H *)		| 164 => (pr "AND A,H\n" ; op_AND_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* AND A, L *)		| 165 => (pr "AND A,L\n" ; op_AND_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* AND A, (HL) *)	| 166 => (pr "AND A,(HL)\n" ; HL2IX (op_AND_r_ptr (#A, putA)) (cs, bs))
(* AND A, A *)		| 167 => (pr "AND A,A\n" ; op_AND_r_r (#A, putA) (#A) (cs, bs)) 
(* XOR A, B *)		| 168 => (pr "XOR A,B\n" ; op_XOR_r_r (#A, putA) (#B) (cs, bs)) 
(* XOR A, C *)		| 169 => (pr "XOR A,C\n" ; op_XOR_r_r (#A, putA) (#C) (cs, bs)) 
(* XOR A, D *)		| 170 => (pr "XOR A,D\n" ; op_XOR_r_r (#A, putA) (#D) (cs, bs)) 
(* XOR A, E *)		| 171 => (pr "XOR A,E\n" ; op_XOR_r_r (#A, putA) (#E) (cs, bs)) 
(* XOR A, H *)		| 172 => (pr "XOR A,H\n" ; op_XOR_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* XOR A, L *)		| 173 => (pr "XOR A,L\n" ; op_XOR_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* XOR A, (HL) *)	| 174 => (pr "XOR A,(HL)\n" ; HL2IX (op_XOR_r_ptr (#A, putA)) (cs, bs))
(* XOR A, A *)		| 175 => (pr "XOR A,A\n" ; op_XOR_r_r (#A, putA) (#A) (cs, bs)) 
(* OR A, B *)		| 176 => (pr "OR A,B\n" ; op_OR_r_r (#A, putA) (#B) (cs, bs)) 
(* OR A, C *)		| 177 => (pr "OR A,C\n" ; op_OR_r_r (#A, putA) (#C) (cs, bs)) 
(* OR A, D *)		| 178 => (pr "OR A,D\n" ; op_OR_r_r (#A, putA) (#D) (cs, bs)) 
(* OR A, E *)		| 179 => (pr "OR A,E\n" ; op_OR_r_r (#A, putA) (#E) (cs, bs)) 
(* OR A, H *)		| 180 => (pr "OR A,H\n" ; op_OR_r_r (#A, putA) (rplGetH cs) (cs, bs)) 
(* OR A, L *)		| 181 => (pr "OR A,L\n" ; op_OR_r_r (#A, putA) (rplGetL cs) (cs, bs)) 
(* OR A, (HL) *)	| 182 => (pr "OR A,(HL)\n" ; HL2IX (op_OR_r_ptr (#A, putA)) (cs, bs))
(* OR A, A *)		| 183 => (pr "OR A,A\n" ; op_OR_r_r (#A, putA) (#A) (cs, bs)) 
(* CP A, B *)		| 184 => (pr "CP A,B\n" ; op_CP_r_r (#A) (#B) (cs, bs)) 
(* CP A, C *)		| 185 => (pr "CP A,C\n" ; op_CP_r_r (#A) (#C) (cs, bs)) 
(* CP A, D *)		| 186 => (pr "CP A,D\n" ; op_CP_r_r (#A) (#D) (cs, bs)) 
(* CP A, E *)		| 187 => (pr "CP A,E\n" ; op_CP_r_r (#A) (#E) (cs, bs)) 
(* CP A, H *)		| 188 => (pr "CP A,H\n" ; op_CP_r_r (#A) (rplGetH cs) (cs, bs)) 
(* CP A, L *)		| 189 => (pr "CP A,L\n" ; op_CP_r_r (#A) (rplGetL cs) (cs, bs)) 
(* CP A, (HL) *)	| 190 => (pr "CP A,(HL)\n" ; HL2IX (op_CP_r_ptr (#A)) (cs, bs))
(* CP A, A *)		| 191 => (pr "CP A,A\n" ; op_CP_r_r (#A) (#A) (cs, bs)) 
(* RET NZ     *)	| 192 => (pr "RET NZ\n" ; op_RET_cond (fn cs => Bit.isNotSet (getFlag_Z cs)) (cs, bs)) 
(* POP BC   *)		| 193 => (pr "POP BC\n" ; op_POP_rr putBC (cs, bs)) 
(* JP NZ,nn     *)	| 194 => (pr "JP NZ,nn\n" ; op_JP_cond (fn cs => Bit.isNotSet (getFlag_Z cs)) (cs, bs)) 
(* JP nn      *)	| 195 => (pr "JP nn\n" ; op_JP (cs, bs)) 
(* CALL NZ     *)	| 196 => (pr "CALL NZ\n" ; op_CALL_cond (fn cs => Bit.isNotSet (getFlag_Z cs)) (cs, bs)) 
(* PUSH BC   *)		| 197 => (pr "PUSH BC\n" ; op_PUSH_rr getBC (cs, bs)) 
(* ADD A, n *)		| 198 => (pr "ADD A,n\n" ; op_ADD_r_n (#A, putA) (cs, bs)) 
(* RST 0h      *)	| 199 => (pr "RST 0h\n" ; op_RST_adr 0 (cs, bs)) 
(* RET Z      *)	| 200 => (pr "RET Z\n" ; op_RET_cond (fn cs => Bit.isSet (getFlag_Z cs)) (cs, bs)) 
(* RET        *)	| 201 => (pr "RET\n" ; op_RET (cs, bs)) 
(* JP Z,nn      *)	| 202 => (pr "JP Z,nn\n" ; op_JP_cond (fn cs => Bit.isSet (getFlag_Z cs)) (cs, bs)) 
(* Prefix : CB *)	| 203 => (pr "CB:\n" ; oOCF choose_opcode_CB (cs, bs)) 
(* CALL Z      *)	| 204 => (pr "CALL Z\n" ; op_CALL_cond (fn cs => Bit.isSet (getFlag_Z cs)) (cs, bs)) 
(* CALL nn      *)	| 205 => (pr "CALL nn\n" ; op_CALL (cs, bs)) 
(* ADC A, n *)		| 206 => (pr "ADC A,n\n" ; op_ADC_r_n (#A, putA) (cs, bs)) 
(* RST 8h      *)	| 207 => (pr "RST 8h\n" ; op_RST_adr 8 (cs, bs)) 
(* RET NC     *)	| 208 => (pr "RET NC\n" ; op_RET_cond (fn cs => Bit.isNotSet (getFlag_C cs)) (cs, bs)) 
(* POP DE   *)		| 209 => (pr "POP DE\n" ; op_POP_rr putDE (cs, bs)) 
(* JP NC,nn     *)	| 210 => (pr "JP NC,nn\n" ; op_JP_cond (fn cs => Bit.isNotSet (getFlag_C cs)) (cs, bs)) 
(* OUT (n),A     *)	| 211 => (pr "OUT(n),A\n" ; op_OUT_ptrn_A (cs, bs)) 
(* CALL NC     *)	| 212 => (pr "CALL NC\n" ; op_CALL_cond (fn cs => Bit.isNotSet (getFlag_C cs)) (cs, bs)) 
(* PUSH DE   *)		| 213 => (pr "PUSH DE\n" ; op_PUSH_rr getDE (cs, bs)) 
(* SUB A, n *)		| 214 => (pr "SUB A,n\n" ; op_SUB_r_n (#A, putA) (cs, bs)) 
(* RST 10h      *)	| 215 => (pr "RST 10h\n" ; op_RST_adr 16 (cs, bs)) 
(* RET C      *)	| 216 => (pr "RET C\n" ; op_RET_cond (fn cs => Bit.isSet (getFlag_C cs)) (cs, bs)) 
(* EXX        *)	| 217 => (pr "EXX\n" ; op_EXX  (cs, bs)) 
(* JP C,nn      *)	| 218 => (pr "JP C,nn\n" ; op_JP_cond (fn cs => Bit.isSet (getFlag_C cs)) (cs, bs)) 
(* IN A,(n)     *)	| 219 => (pr "IN A,(n)\n" ; op_IN_A_ptrn (cs, bs)) 
(* CALL C      *)	| 220 => (pr "CALL C\n" ; op_CALL_cond (fn cs => Bit.isSet (getFlag_C cs)) (cs, bs)) 

(* Prefix : DD *)	| 221 => (pr "DD:\n" ; oOCF choose_opcode (putHLas asIX cs, bs)) 

(* SBC A, n *)		| 222 => (pr "SBC A,n\n" ; op_SBC_r_n (#A, putA) (cs, bs)) 
(* RST 18h      *)	| 223 => (pr "RST 18h\n" ; op_RST_adr 24 (cs, bs)) 
(* RET PO     *)	| 224 => (pr "RET PO\n" ; op_RET_cond (fn cs => Bit.isNotSet (getFlag_PV cs)) (cs, bs)) 
(* POP HL   *)		| 225 => (pr "POP HL\n" ; op_POP_rr (rplPutHL cs) (cs, bs)) 
(* JP PO,nn     *)	| 226 => (pr "JP PO,nn\n" ; op_JP_cond (fn cs => Bit.isNotSet (getFlag_PV cs)) (cs, bs)) 
(* EX (SP),HL   *)	| 227 => (pr "EX (SP),HL\n" ; op_EX_SP_rr (rplGetHL cs, rplPutHL cs) (cs, bs)) 
(* CALL PO     *)	| 228 => (pr "CALL PO\n" ; op_CALL_cond (fn cs => Bit.isNotSet (getFlag_PV cs)) (cs, bs)) 
(* PUSH HL   *)		| 229 => (pr "PUSH HL\n" ; op_PUSH_rr (rplGetHL cs) (cs, bs)) 
(* AND A, n *)		| 230 => (pr "AND A,n\n" ; op_AND_r_n (#A, putA) (cs, bs)) 
(* RST 20h      *)	| 231 => (pr "RST 20h\n" ; op_RST_adr 32 (cs, bs)) 
(* RET PE      *)	| 232 => (pr "RET PE\n" ; op_RET_cond (fn cs => Bit.isSet (getFlag_PV cs)) (cs, bs)) 
(* JP (HL)      *)	| 233 => (pr "JP (HL)\n" ; op_JP_ptr (rplGetHL cs cs) (cs, bs)) (* This time (HL) means HL*) 
(* JP PE,nn      *)	| 234 => (pr "JP PE,nn\n" ; op_JP_cond (fn cs => Bit.isSet (getFlag_PV cs)) (cs, bs)) 
(* EX DE,HL   *)	| 235 => (pr "EX DE,HL\n" ; op_EX_rr_rr (getHL, putHL) (getDE, putDE) (cs, bs)) (* Exception - HL is not trated as IX or IY *)
(* CALL PE      *)	| 236 => (pr "CALL PE\n" ; op_CALL_cond (fn cs => Bit.isSet (getFlag_PV cs)) (cs, bs)) 

(* Prefix : ED *)	| 237 => (pr "ED:\n" ; oOCF choose_opcode_ED (putHLas asHL cs, bs)) 

(* XOR A, n *)		| 238 => (pr "XOR A,n\n" ; op_XOR_r_n (#A, putA) (cs, bs)) 
(* RST 28h      *)	| 239 => (pr "RST 28h\n" ; op_RST_adr 40 (cs, bs)) 
(* RET P      *)	| 240 => (pr "RET P\n" ; op_RET_cond (fn cs => Bit.isNotSet (getFlag_S cs)) (cs, bs)) 
(* POP AF   *)		| 241 => (pr "POP AF\n" ; op_POP_rr putAF (cs, bs)) 
(* JP P,nn      *)	| 242 => (pr "JP P\n" ; op_JP_cond (fn cs => Bit.isNotSet (getFlag_S cs)) (cs, bs)) 
(* DI      *)		| 243 => (pr "DI\n" ; op_DI (cs, bs))
(* CALL P      *)	| 244 => (pr "CALL P\n" ; op_CALL_cond (fn cs => Bit.isNotSet (getFlag_S cs)) (cs, bs)) 
(* PUSH AF   *)		| 245 => (pr "PUSH AF\n" ; op_PUSH_rr getAF (cs, bs)) 
(* OR A, n *)		| 246 => (pr "OR A,n\n" ; op_OR_r_n (#A, putA) (cs, bs)) 
(* RST 30h      *)	| 247 => (pr "RST 30h\n" ; op_RST_adr 48 (cs, bs)) 
(* RET M      *)	| 248 => (pr "RET PM\n" ; op_RET_cond (fn cs => Bit.isSet (getFlag_S cs)) (cs, bs)) 
(* LD SP,HL      *)	| 249 => (pr "LD SP,HL\n" ; op_LD_rr_rr putSP (rplGetHL cs) (cs, bs))
(* JP M,nn      *)	| 250 => (pr "JP PM,nn\n" ; op_JP_cond (fn cs => Bit.isSet (getFlag_S cs)) (cs, bs)) 
(* EI      *)		| 251 => (pr "EI\n" ; op_EI (cs, bs))
(* CALL M      *)	| 252 => (pr "CALL PM\n" ; op_CALL_cond (fn cs => Bit.isSet (getFlag_S cs)) (cs, bs)) 

(* Prefix : FD *)	| 253 => (pr "FD:\n" ; oOCF choose_opcode (putHLas asIY cs, bs)) 

(* CP A, n *)		| 254 => (pr "CP A,n\n" ; op_CP_r_n (#A) (cs, bs)) 
(* RST 38h      *)	| 255 => (pr "RST 38h\n" ; op_RST_adr 56 (cs, bs)) 
			| z   => (pr (unknown_opcode_string z cs) ; 
				raise UNKNOWN_OPCODE {opcode = opcode, PC = (#PC cs)})
		end		
	;

        val initial_state : device_state = DS(cpu_initial_state, next_instruction)

	fun readStream str =
		let
			val readReg8 = fn (_ : unit) =>
				let
					val w = valOf (BinIO.input1 str);
					val v = Word8.toInt w;
					val r = Register8.fromInt v;
				in
					r
				end
			;
			val readReg16 = fn (_ : unit) =>
				let
					val w = valOf (BinIO.input1 str);
					val v = Word8.toInt w;
					val r = Register8.fromInt v;
					val w1 = valOf (BinIO.input1 str);
					val v1 = Word8.toInt w1;
					val r1 = Register8.fromInt v1;
				in
					Register16.fromHalves (Register8.asBitfield r, Register8.asBitfield r1)
				end
			;
			fun convIM r cs =
				let
					val i = Register8.asUnsigned r;
				in
					if i > 2 then 
						raise Fail ("Invalid interrupt mode during reading snapshot")
					else
						putIM i cs
				end
			;
			fun convIFFs r cs =
				(
					(putIFF1 (Bit.asBool (Register8.getBit 1 r)))
					o (putIFF2 (Bit.asBool (Register8.getBit 2 r)))
				) cs
			;
			val cs = putI (readReg8 () ) cpu_initial_state;
			val _ = pr (cpuAsString cs) ;
			val cs = putHL' (readReg16 () ) cs;
			val cs = putDE' (readReg16 () ) cs;
			val cs = putBC' (readReg16 () ) cs;
			val cs = putAF' (readReg16 () ) cs;
			val cs = putHL (readReg16 () ) cs;
			val cs = putDE (readReg16 () ) cs;
			val cs = putBC (readReg16 () ) cs;
			val cs = putIY (readReg16 () ) cs;
			val cs = putIX (readReg16 () ) cs;
			val cs = convIFFs (readReg8 () ) cs;
			val cs = putR (readReg8 () ) cs;
			val cs = putAF (readReg16 () ) cs;
			val cs = putSP (readReg16 () ) cs;
			val cs = convIM (readReg8 () ) cs;
			val _  = readReg8 ();
			val _ = pr (cpuAsString cs) ;
		in
			DS(cs, op_RET)
		end
	;

end;

