fun TwoExp (n: int) = 
let
	fun t acc k = if k <= 0 then acc else t (acc*2) (k-1);
in
	t 1 n
end;


functor BitfieldOnList(Bit : BIT) : BITFIELD = 
struct
	structure Bit = Bit;
	type bitfield = Bit.bit list;
	exception BITFIELD_ERROR of string;

	fun replace 0 e (a::tl) = e::tl
	  | replace n e (a::tl) = 
		if n>0 then
			a::(replace (n-1) e tl)
		else
			raise BITFIELD_ERROR("Trying to replace element with negative index")
	  | replace _ _ _ = raise BITFIELD_ERROR("Trying to replace nonexisting element")
	;

	val maxWidth : int option = NONE;
	fun nZeros (n : int) = List.tabulate (n, fn _ => Bit.zero);
	fun new  (w : int) =
		if w>0 then
			nZeros w
		else
			raise BITFIELD_ERROR("Wrong width of bitfield")
	;
	fun width (bf: bitfield)  = List.length bf;

	fun inBounds (i : int) (bf: bitfield) = i >= 0 andalso i < width bf;

	fun maxSigned (bf : bitfield) = (TwoExp ((width bf) - 1)) - 1;
	fun maxUnsigned (bf : bitfield) = TwoExp (width bf) - 1;
	fun minSigned (bf : bitfield) = ~ (TwoExp ((width bf - 1)));
	fun minUnsigned (bf : bitfield)  = 0;
	fun getBit (n : int) (bf : bitfield) =
		if not (inBounds n bf) then
			raise BITFIELD_ERROR("getBit out of bounds")
		else
			List.nth (bf, n)
	;
	fun putBit (n: int) (b : Bit.bit) (bf : bitfield) =
		if not (inBounds n bf) then
			raise BITFIELD_ERROR("getBit out of bounds")
		else
			replace n b bf
	;
	fun setBit (n : int) (bf : bitfield) =
		if not (inBounds n bf) then
			raise BITFIELD_ERROR("setBit")
		else
			replace n Bit.one bf
	;
	fun resetBit (n : int) (bf : bitfield) =
		if not (inBounds n bf) then
			raise BITFIELD_ERROR("resetBit")
		else
			replace n Bit.zero bf
	;
	fun negateBit (n : int) (bf : bitfield) =
		if not (inBounds n bf) then
			raise BITFIELD_ERROR("negateBit")
		else
			replace n (Bit.negate (getBit n bf)) bf
	;
	fun isSet (n : int) (bf : bitfield) =
		Bit.isSet (getBit n bf)
	;
	fun isNotSet (n : int) (bf : bitfield) = not (isSet n bf);
	fun asSigned (bf : bitfield) =
		let
			fun t n acc (a::nil) = if Bit.isSet a then acc - n else acc
			  | t n acc (a::tl) = 
				t (n*2) 
				(if (Bit.isSet a) then (acc + n) else acc)
				tl
			  | t _ _ _ = raise BITFIELD_ERROR("Error during conversion to signed");
		in
			t 1 0 bf
		end
	;
	fun asUnsigned (bf : bitfield) =
		let
			fun t n acc [] = acc
			  | t n acc (a::tl) = 
				t (n*2) 
				(if (Bit.isSet a) then (acc + n) else acc)
				tl
			;
		in
			t 1 0 bf
		end
	;
	fun map (bf : bitfield) (f : Bit.bit -> Bit.bit) : bitfield =
		let
			fun t [] g = []
			  | t (a::tl) g = (g a)::(t tl g)
			;
		in
			t bf f
		end
	;
	fun negateBits (bf : bitfield) = map bf (Bit.negate);
	fun addBits (carry : Bit.bit, bf1: bitfield, bf2: bitfield) : (bitfield * bitfield) = 
		let
			fun add [] [] (carry : Bit.bit) = ([], [])
			  | add (a1::tl1) (a2::tl2) carry =
				let
					val (v, c) = Bit.addWithCarry (a1, a2, carry);
					val (lv, lc) = add tl1 tl2 c;
				in
					(v::lv, c::lc)
				end
			 | add _ _ _ = raise BITFIELD_ERROR ("unexpected error during adding bitfields");
			;
		in
			if width bf1 <> width bf2 then
				raise BITFIELD_ERROR("adding bitfields of inequal width")
			else
				add bf1 bf2 carry
		end
	;
	fun subtractBits (borrow : Bit.bit, bf1: bitfield, bf2: bitfield) : (bitfield * bitfield) = 
		let
			fun subtract [] [] (borrow : Bit.bit) = ([], [])
			  | subtract (a1::tl1) (a2::tl2) borrow =
				let
					val (v, c) = Bit.subtractWithBorrow (a1, a2, borrow);
					val (lv, lc) = subtract tl1 tl2 c;
				in
					(v::lv, c::lc)
				end
			  | subtract _ _ _ = raise BITFIELD_ERROR ("unexpected error during subtracting bitfields")
			;
		in
			if width bf1 <> width bf2 then
				raise BITFIELD_ERROR("subtracting bitfields of inequal width")
			else
				subtract bf1 bf2 borrow
		end
	;
	fun u2Bits (bf : bitfield) = 
		let
			val one = setBit 0 (new (width bf));
			val (v, _) = addBits (Bit.zero, (negateBits bf), one);
		in
			v
		end
	;
	fun fromInt (i : int) (w : int) : bitfield =
		let
			fun conv k r 0 = (nil, k)
			  | conv k r n =
				let
					val (l, v) = conv k (r*2) (n-1)
				in
					if (v >= r) then
						(Bit.one::l, (v - r))
					else
						(Bit.zero::l, v)
				end
			;
			val (bf, _) = conv (abs(i)) 1 w;
		in
			if i < minSigned bf orelse i>maxUnsigned bf then
				raise BITFIELD_ERROR("Error during conversion from int")
			else
				if (i>=0) then bf else u2Bits bf
		end
	;
	fun map2 (bf1: bitfield , bf2 : bitfield) (f : Bit.bit * Bit.bit -> Bit.bit) = 
		let 
			fun t [] _ f = []
			  | t _ [] f = []
			  | t (a1::tl1) (a2::tl2) f = (f (a1,a2))::(t tl1 tl2 f)
			;
		in
			if (width bf1) <> (width bf2) then
				raise BITFIELD_ERROR("Mapping bitfields of inequal length")
			else
				t bf1 bf2 f
		end
	;
	fun andBits (bf1: bitfield , bf2 : bitfield) = map2 (bf1, bf2) (Bit.andBits);
	fun orBits (bf1: bitfield , bf2 : bitfield) = map2 (bf1, bf2) (Bit.orBits);
	fun xorBits (bf1: bitfield , bf2 : bitfield) = map2 (bf1, bf2) (Bit.xorBits);

	fun rolBits (i : int) (bf : bitfield) =
		if i>0 andalso i<(width bf) then
			List.take (List.drop (bf @ bf, (width bf) - i), width bf)
		else
			raise BITFIELD_ERROR ("rolBits out of bounds")
	;
	fun rorBits (i : int) (bf : bitfield) =
		if i>0 andalso i<(width bf) then
			List.take (List.drop (bf @ bf, i), width bf)
		else
			raise BITFIELD_ERROR ("rorBits out of bounds")
	;

	fun countOnes (bf : bitfield) = 
		let
			fun c acc [] = acc
			  | c acc (a :: tl) = c (acc + (Bit.asInt a)) tl
			;
		in
			c 0 bf
		end
	;

	fun subField (orig : int) (len : int) (bf : bitfield) : bitfield = 
		if orig + len > (width bf) then
			raise BITFIELD_ERROR("Trying to make too big subfield")
		else
			List.take (List.drop (bf, orig), len)
	;

	fun append (bf1 : bitfield, bf2 : bitfield) : bitfield = bf1 @ bf2;
	fun fromBit (b : Bit.bit) : bitfield = [b];
	fun asString (bf : bitfield) =
		let
			fun g [] = ""
			  | g (a::tl) = (g tl) ^ (Bit.asString a)
			;
		in
			"[" ^ (g bf) ^ "]"
		end
	;
end;
